<?php
/**
 * DigiMaster Telegram Proxy (v3)
 *
 * این فایل را کنار wp-load.php قرار دهید و آدرس آن را در config.php ربات ست کنید.
 */

if (!defined('ABSPATH')) {
    // سعی کن وردپرس را لود کنی (کنار wp-load.php قرار بگیرد)
    $wp_load = __DIR__ . '/wp-load.php';
    if (!file_exists($wp_load)) {
        http_response_code(500);
        echo json_encode(['ok'=>false,'error'=>['code'=>'no_wp','message'=>'wp-load.php not found beside proxy']]);
        exit;
    }
    require_once $wp_load;
}

// پاسخ JSON استاندارد
function dgm_proxy_json($data, $code = 200) {
    if (!headers_sent()) {
        http_response_code($code);
        header('Content-Type: application/json; charset=utf-8');
    }
    echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

// خطا
function dgm_proxy_error($code, $message, $extra = [], $http = 400) {
    $payload = [
        'ok'    => false,
        'error' => array_merge([
            'code'    => $code,
            'message' => $message,
        ], (array)$extra),
    ];
    dgm_proxy_json($payload, $http);
}

// فقط POST و JSON
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    dgm_proxy_error('bad_method', 'POST required', [], 405);
}

$raw = file_get_contents('php://input');
$data = json_decode($raw, true);
if (!is_array($data)) {
    dgm_proxy_error('bad_json', 'Invalid JSON body');
}

// توکن ایمنی ساده – حتماً در config.php ربات و وردپرس همان را ست کنید
$expected_token = defined('DGM_PROXY_TOKEN') ? DGM_PROXY_TOKEN : 'CHANGE_ME_SECRET';
$token          = isset($data['token']) ? (string)$data['token'] : '';
if (!$token || $token !== $expected_token) {
    dgm_proxy_error('unauthorized', 'Invalid proxy token', [], 401);
}

$action  = isset($data['action'])  ? (string)$data['action']  : '';
$payload = isset($data['payload']) ? (array)$data['payload'] : [];

if ($action === '') {
    dgm_proxy_error('no_action', 'Action is required');
}

// اطمینان از اینکه خروجی وردپرس به زبان فارسی برنمی‌گردد
if (function_exists('load_default_textdomain')) {
    load_default_textdomain();
}

// ست کردن کاربر فعلی به ادمین (برای دسترسی manage_options / manage_woocommerce)
if (!is_user_logged_in()) {
    $admins = get_users(['role' => 'administrator', 'number' => 1]);
    if (!empty($admins)) {
        wp_set_current_user($admins[0]->ID);
    }
}

try {
    switch ($action) {

        /* ======================== لیست دسته‌های ووکامرس ======================== */
        case 'wc_categories':
        {
            if (!taxonomy_exists('product_cat')) {
                dgm_proxy_error('no_wc', 'taxonomy product_cat not found (WooCommerce active?)');
            }

            $search = isset($payload['search']) ? sanitize_text_field($payload['search']) : '';
            $args = [
                'taxonomy'   => 'product_cat',
                'hide_empty' => false,
                'number'     => 50,
                'orderby'    => 'name',
                'order'      => 'ASC',
            ];
            if ($search !== '') {
                // تطبیق نام
                $args['name__like'] = $search;
            }

            $terms = get_terms($args);
            if (is_wp_error($terms)) {
                dgm_proxy_error('terms_error', $terms->get_error_message());
            }

            $out = [];
            foreach ($terms as $t) {
                $id   = (int)$t->term_id;
                $name = $t->name;
                $parent = (int)$t->parent;
                $parts = [$name];
                $seen  = [$id];
                while ($parent) {
                    $pt = get_term($parent, 'product_cat');
                    if (is_wp_error($pt) || !$pt) break;
                    if (in_array((int)$pt->term_id, $seen, true)) break; // جلوگیری از حلقه
                    array_unshift($parts, $pt->name);
                    $seen[] = (int)$pt->term_id;
                    $parent = (int)$pt->parent;
                }
                $full = implode(' > ', $parts);
                $out[] = [
                    'id'        => $id,
                    'name'      => $name,
                    'full_name' => $full,
                    'parent'    => (int)$t->parent,
                    'slug'      => $t->slug,
                ];
            }

            dgm_proxy_json([
                'ok'         => true,
                'categories' => $out,
            ]);
        }
        break;

        /* ======================== دیجی‌کالا – ایمپورت تکی ======================== */
        case 'digikala_single':
        {
            if (!function_exists('dm_import_digikala_single_core')) {
                dgm_proxy_error('missing_importer', 'dm_import_digikala_single_core not available');
            }

            $idOrUrl = isset($payload['id_or_url']) ? trim((string)$payload['id_or_url']) : '';
            $opts    = isset($payload['opts']) ? (array)$payload['opts'] : [];

            if ($idOrUrl === '') {
                dgm_proxy_error('bad_input', 'شناسه یا لینک دیجی‌کالا وارد نشده است');
            }

            // استفاده از سرویس رسمی Digimaster (در صورت موجود بودن فیلتر)
            if (has_filter('dgm/service/import_digikala_single')) {
                $result = apply_filters('dgm/service/import_digikala_single', null, [
                    'id_or_url' => $idOrUrl,
                    'opts'      => $opts,
                ]);
            } else {
                $result = dm_import_digikala_single_core($idOrUrl, $opts);
            }

            // اگر WP_Error برگشته
            if (is_wp_error($result)) {
                $code = $result->get_error_code() ?: 'import_failed';
                $msg  = $result->get_error_message() ?: 'خطای نامشخص در ایمپورت دیجی‌کالا';

                // اگر no_pid بود، یک تلاش اضافه برای یافتن محصول بر اساس متای digikala_id انجام بده
                if ($code === 'no_pid') {
                    $dkp = null;
                    if (preg_match('~digikala\.com/.*/dkp-(\d+)~i', $idOrUrl, $m)) {
                        $dkp = $m[1];
                    } elseif (ctype_digit($idOrUrl)) {
                        $dkp = $idOrUrl;
                    }

                    if ($dkp) {
                        global $wpdb;
                        $pid = (int) $wpdb->get_var(
                            $wpdb->prepare(
                                "SELECT post_id FROM {$wpdb->postmeta}
                                 WHERE meta_key = %s AND meta_value = %s
                                 ORDER BY post_id DESC
                                 LIMIT 1",
                                'digikala_id',
                                $dkp
                            )
                        );

                        if ($pid > 0) {
                            dgm_proxy_json([
                                'ok'         => true,
                                'source'     => 'digikala',
                                'product_id' => $pid,
                                'meta'       => ['dkp' => (int)$dkp, 'fallback' => 'meta_lookup'],
                            ]);
                        }
                    }
                }

                dgm_proxy_error($code, $msg);
            }

            // اگر خروجی آرایه باشد (مطابق dm_import_digikala_single_core)
            if (is_array($result) && !empty($result['ok']) && !empty($result['product_id'])) {
                dgm_proxy_json([
                    'ok'         => true,
                    'source'     => 'digikala',
                    'product_id' => (int)$result['product_id'],
                    'meta'       => isset($result['meta']) ? $result['meta'] : [],
                ]);
            }

            // هر چیز دیگری را خطا بگیر
            dgm_proxy_error('unexpected_result', 'خروجی ایمپورتر دیجی‌کالا غیرمنتظره است');
        }
        break;

        /* ======================== تکنولایف – ایمپورت تکی ======================== */
        case 'techno_single':
        {
            if (!function_exists('tni_create_from_url')) {
                dgm_proxy_error('missing_importer', 'tni_create_from_url not available');
            }

            $url   = isset($payload['url']) ? trim((string)$payload['url']) : '';
            $catId = isset($payload['category']) ? (int)$payload['category'] : 0;

            if ($url === '' || $catId <= 0) {
                dgm_proxy_error('bad_input', 'لینک تکنولایف و دسته مقصد الزامی است');
            }

            $res = tni_create_from_url($url, $catId);
            if (is_wp_error($res)) {
                dgm_proxy_error(
                    $res->get_error_code() ?: 'import_failed',
                    $res->get_error_message() ?: 'خطای ایمپورت تکنولایف'
                );
            }

            $pid = isset($res['product_id']) ? (int)$res['product_id'] : 0;
            if ($pid <= 0) {
                dgm_proxy_error('no_pid', 'شناسه محصول ووکامرس برنگشت (تکنولایف)');
            }

            dgm_proxy_json([
                'ok'         => true,
                'source'     => 'technolife',
                'product_id' => $pid,
            ]);
        }
        break;

        /* ======================== تکنولایف – ایمپورت دسته‌ای ======================== */
        case 'techno_bulk':
        {
            if (!function_exists('tni_create_from_url')) {
                dgm_proxy_error('missing_importer', 'tni_create_from_url not available');
            }

            $url        = isset($payload['url']) ? trim((string)$payload['url']) : '';
            $pages      = isset($payload['pages']) ? (int)$payload['pages'] : 1;
            $catId      = isset($payload['category']) ? (int)$payload['category'] : 0;
            $stock_only = !empty($payload['stock_only']) ? 1 : 0;

            if ($url === '' || $catId <= 0) {
                dgm_proxy_error('bad_input', 'لینک دسته/جستجوی تکنولایف و دسته مقصد الزامی است');
            }
            if ($pages < 1) $pages = 1;
            if ($pages > 20) $pages = 20; // محدودیت منطقی

            $all_links = [];
            $stop_when_empty = true;
            for ($pg = 1; $pg <= $pages; $pg++) {
                $api = add_query_arg([
                    'url'             => $url,
                    'pg'              => $pg,
                    'pages'           => 1,
                    'stock'           => $stock_only ? 1 : 0,
                    'stop_when_empty' => $stop_when_empty ? 1 : 0,
                    'debug'           => 0,
                ], 'https://majazi.top/v2/techno.php');

                $res = wp_safe_remote_get($api, [
                    'timeout' => 30,
                    'sslverify' => true,
                    'headers' => ['Accept' => 'application/json'],
                ]);

                if (is_wp_error($res)) {
                    dgm_proxy_error('remote_error', 'خطای اتصال به وب‌سرویس تکنولایف: '.$res->get_error_message(), ['api'=>$api], 502);
                }

                $code = wp_remote_retrieve_response_code($res);
                $body = wp_remote_retrieve_body($res);
                $json = json_decode($body, true);

                if ($code !== 200 || !is_array($json) || empty($json['ok'])) {
                    dgm_proxy_error('remote_error', 'پاسخ نامعتبر از وب‌سرویس تکنولایف', [
                        'code' => $code,
                        'body' => mb_substr($body, 0, 800),
                        'api'  => $api,
                    ], 502);
                }

                $products = $json['products'] ?? [];
                $collector = [];
                $collect = function($node) use (&$collect, &$collector){
                    if (is_string($node)) {
                        if (preg_match('~https?://[^"\s]+/product-(\d+)(/|\b)~i', $node)) $collector[] = $node;
                    } elseif (is_array($node)) {
                        foreach ($node as $v) $collect($v);
                    }
                };
                $collect($products);

                $links = array_values(array_unique(array_map(function($u){
                    $u = preg_replace('~//www\.technolife\.ir~i', '//www.technolife.com', $u);
                    return esc_url_raw($u);
                }, $collector)));

                if (empty($links) && $stop_when_empty) {
                    // اگر این صفحه چیزی نداشت و stop_when_empty فعال است، حلقه را قطع کن
                    break;
                }

                $all_links = array_merge($all_links, $links);
            }

            $all_links = array_values(array_unique(array_filter($all_links)));
            $total_links = count($all_links);
            if ($total_links === 0) {
                dgm_proxy_error('no_links', 'هیچ لینک محصولی از صفحات تکنولایف یافت نشد.');
            }

            $created = 0;
            $skipped = 0;
            $errors  = 0;
            $error_samples = [];

            foreach ($all_links as $lnk) {
                $canon = function_exists('tni_canonical_url') ? tni_canonical_url($lnk) : esc_url_raw($lnk);
                if (!$canon) {
                    $skipped++;
                    continue;
                }

                // duplicate check مشابه ajax_techno_bulk_import_one
                $existing = wc_get_products([
                    'limit'        => 1,
                    'return'       => 'ids',
                    'meta_key'     => 'technolife_link',
                    'meta_value'   => $canon,
                    'meta_compare' => '=',
                ]);
                if (!empty($existing)) {
                    $skipped++;
                    continue;
                }

                $res = tni_create_from_url($canon, $catId);
                if (is_wp_error($res)) {
                    $errors++;
                    if (count($error_samples) < 5) {
                        $error_samples[] = $res->get_error_message();
                    }
                    continue;
                }

                $pid = isset($res['product_id']) ? (int)$res['product_id'] : 0;
                if ($pid > 0) {
                    $created++;
                } else {
                    $errors++;
                }
            }

            dgm_proxy_json([
                'ok'          => true,
                'source'      => 'technolife',
                'total_links' => $total_links,
                'created'     => $created,
                'skipped'     => $skipped,
                'errors'      => $errors,
                'error_samples' => $error_samples,
            ]);
        }
        break;

        /* ======================== ترب – ایمپورت تکی ======================== */
        case 'torob_single':
        {
            if (!class_exists('DM_Torob_Standalone')) {
                dgm_proxy_error('missing_importer', 'DM_Torob_Standalone not available');
            }

            $prk   = isset($payload['prk']) ? trim((string)$payload['prk']) : '';
            $url   = isset($payload['url']) ? trim((string)$payload['url']) : '';
            $catId = isset($payload['category']) ? (int)$payload['category'] : 0;

            if ($catId <= 0) {
                dgm_proxy_error('bad_input', 'دسته مقصد برای ایمپورت ترب الزامی است');
            }
            if ($prk === '' && $url === '') {
                dgm_proxy_error('bad_input', 'حداقل یکی از PRK یا لینک ترب باید ارسال شود');
            }

            // اگر PRK خالی بود، سعی کن از روی URL استخراجش کنی با کمک Reflection روی متد خصوصی extract_prk
            if ($prk === '' && $url !== '') {
                try {
                    $refClass = new ReflectionClass('DM_Torob_Standalone');
                    if ($refClass->hasMethod('extract_prk')) {
                        $m = $refClass->getMethod('extract_prk');
                        $m->setAccessible(true);
                        $prk = (string)$m->invoke(null, $url);
                    }
                } catch (Throwable $e) {
                    // نادیده بگیر، اگر نشد بعداً خطا می‌دهیم
                }
            }

            if ($prk === '') {
                dgm_proxy_error('no_prk', 'PRK از لینک ترب قابل استخراج نیست');
            }

            // حالا با Reflection به متدهای خصوصی fetch_details و upsert_wc_product دسترسی می‌گیریم
            try {
                $refClass = new ReflectionClass('DM_Torob_Standalone');

                if (!$refClass->hasMethod('fetch_details') || !$refClass->hasMethod('upsert_wc_product')) {
                    dgm_proxy_error('missing_methods', 'متدهای داخلی ترب برای ایمپورت قابل دسترس نیستند');
                }

                $mFetch = $refClass->getMethod('fetch_details');
                $mFetch->setAccessible(true);
                $details = $mFetch->invoke(null, $prk);
                if (is_wp_error($details)) {
                    dgm_proxy_error(
                        $details->get_error_code() ?: 'import_failed',
                        $details->get_error_message() ?: 'خطای دریافت جزئیات از ترب'
                    );
                }

                $mUpsert = $refClass->getMethod('upsert_wc_product');
                $mUpsert->setAccessible(true);
                $pid = $mUpsert->invoke(null, $details, $catId);
                if (is_wp_error($pid)) {
                    dgm_proxy_error(
                        $pid->get_error_code() ?: 'import_failed',
                        $pid->get_error_message() ?: 'خطای ساخت محصول ترب'
                    );
                }

                $pid = (int)$pid;
                if ($pid <= 0) {
                    dgm_proxy_error('no_pid', 'شناسه محصول ووکامرس برنگشت (ترب)');
                }

                dgm_proxy_json([
                    'ok'         => true,
                    'source'     => 'torob',
                    'product_id' => $pid,
                    'meta'       => ['prk' => $prk],
                ]);

            } catch (Throwable $e) {
                dgm_proxy_error('exception', $e->getMessage());
            }
        }
        break;

        /* ======================== ترب – ایمپورت دسته‌ای ======================== */
        case 'torob_bulk':
        {
            if (!class_exists('DM_Torob_Standalone')) {
                dgm_proxy_error('missing_importer', 'DM_Torob_Standalone not available');
            }

            $cat_url = isset($payload['category_url']) ? esc_url_raw((string)$payload['category_url']) : '';
            $pages   = isset($payload['pages']) ? (int)$payload['pages'] : 1;
            $wcCat   = isset($payload['category']) ? (int)$payload['category'] : 0;

            if ($wcCat <= 0) {
                dgm_proxy_error('bad_input', 'دسته مقصد برای ایمپورت دسته‌ای ترب الزامی است');
            }
            if ($cat_url === '') {
                dgm_proxy_error('bad_input', 'لینک دسته‌بندی ترب الزامی است');
            }
            if ($pages < 1) $pages = 1;
            if ($pages > 20) $pages = 20;

            try {
                $refClass = new ReflectionClass('DM_Torob_Standalone');

                if (!$refClass->hasMethod('extract_category_params')
                    || !$refClass->hasMethod('call_proxy_service')
                    || !$refClass->hasMethod('fetch_details')
                    || !$refClass->hasMethod('upsert_wc_product')
                    || !$refClass->hasMethod('find_product_by_prk')) {
                    dgm_proxy_error('missing_methods', 'متدهای لازم برای ایمپورت دسته‌ای ترب در دسترس نیستند');
                }

                $mParams = $refClass->getMethod('extract_category_params');
                $mParams->setAccessible(true);
                $params = $mParams->invoke(null, $cat_url);
                if (empty($params['category'])) {
                    dgm_proxy_error('bad_category_url', 'پارامتر category در لینک ترب یافت نشد');
                }

                $mCall   = $refClass->getMethod('call_proxy_service');
                $mCall->setAccessible(true);
                $mFetch  = $refClass->getMethod('fetch_details');
                $mFetch->setAccessible(true);
                $mUpsert = $refClass->getMethod('upsert_wc_product');
                $mUpsert->setAccessible(true);
                $mFind   = $refClass->getMethod('find_product_by_prk');
                $mFind->setAccessible(true);

                $all_prks = [];
                for ($page = 0; $page < $pages; $page++) {
                    $payloadSearch = [
                        'fn'            => 'search',
                        'category'      => $params['category'] ?? '',
                        'category_name' => $params['category_name'] ?? '',
                        'brand'         => $params['brand'] ?? '',
                        'brand_name'    => $params['brand_name'] ?? '',
                        'size'          => 24,
                        'sort'          => 'popularity',
                        'page'          => $page,
                    ];

                    $data = $mCall->invoke(null, $payloadSearch);
                    if (is_wp_error($data)) {
                        dgm_proxy_error(
                            $data->get_error_code() ?: 'import_failed',
                            $data->get_error_message() ?: 'خطای دریافت نتایج ترب'
                        );
                    }

                    $prks = [];
                    if (isset($data['results']) && is_array($data['results'])) {
                        foreach ($data['results'] as $it) {
                            if (!empty($it['random_key'])) $prks[] = $it['random_key'];
                        }
                    } elseif (is_array($data)) {
                        foreach ($data as $it) {
                            if (!empty($it['random_key'])) $prks[] = $it['random_key'];
                        }
                    }

                    if (empty($prks)) {
                        // اگر صفحه خالی شد، حلقه را متوقف کن
                        break;
                    }

                    $all_prks = array_merge($all_prks, $prks);
                }

                $all_prks = array_values(array_unique(array_filter($all_prks)));
                $total_prks = count($all_prks);
                if ($total_prks === 0) {
                    dgm_proxy_error('no_prks', 'هیچ PRK‌ از دسته‌بندی ترب یافت نشد');
                }

                $created = 0;
                $updated = 0;
                $errors  = 0;
                $error_samples = [];

                foreach ($all_prks as $prk) {
                    $prk = (string)$prk;
                    if ($prk === '') continue;

                    $existing_before = $mFind->invoke(null, $prk);

                    $details = $mFetch->invoke(null, $prk);
                    if (is_wp_error($details)) {
                        $errors++;
                        if (count($error_samples) < 5) $error_samples[] = $details->get_error_message();
                        continue;
                    }

                    $pid = $mUpsert->invoke(null, $details, $wcCat);
                    if (is_wp_error($pid)) {
                        $errors++;
                        if (count($error_samples) < 5) $error_samples[] = $pid->get_error_message();
                        continue;
                    }

                    $pid = (int)$pid;
                    if ($pid <= 0) {
                        $errors++;
                        continue;
                    }

                    if ($existing_before && (int)$existing_before === $pid) {
                        $updated++;
                    } else {
                        $created++;
                    }
                }

                dgm_proxy_json([
                    'ok'         => true,
                    'source'     => 'torob',
                    'total_prks' => $total_prks,
                    'created'    => $created,
                    'updated'    => $updated,
                    'errors'     => $errors,
                    'error_samples' => $error_samples,
                ]);

            } catch (Throwable $e) {
                dgm_proxy_error('exception', $e->getMessage());
            }
        }
        break;

        /* ======================== باسلام – ایمپورت دسته‌ای غرفه ======================== */
        case 'basalam_bulk':
        {
            if (!function_exists('basalam_http_get_json')) {
                dgm_proxy_error('missing_importer', 'basalam_http_get_json not available');
            }

            $vendor_url  = isset($payload['vendor_url']) ? trim((string)$payload['vendor_url']) : '';
            $cat_id      = isset($payload['category']) ? (int)$payload['category'] : 0;
            $post_status = isset($payload['post_status']) ? sanitize_text_field($payload['post_status']) : 'draft';

            if ($vendor_url === '' || $cat_id <= 0) {
                dgm_proxy_error('bad_input', 'لینک غرفه و دسته مقصد برای باسلام الزامی است');
            }

            $allowed_status = ['publish','draft'];
            if (!in_array($post_status, $allowed_status, true)) {
                $post_status = 'draft';
            }

            // استخراج نام غرفه از URL (همان منطق ایمپورتر اصلی)
            $path = parse_url($vendor_url, PHP_URL_PATH);
            if (!$path || !preg_match('~^/([^/]+)~', $path, $m)) {
                dgm_proxy_error('bad_vendor', 'لینک غرفه باسلام معتبر نیست');
            }
            $vendorName = sanitize_title($m[1]);

            $apiUrl = "https://majazi.top/basalam/shop.php?name={$vendorName}";
            $data   = basalam_http_get_json($apiUrl);

            if (isset($data['error'])) {
                dgm_proxy_error('remote_error', 'خطای دریافت از وب‌سرویس: ' . $data['error']);
            }
            if (empty($data['products']) || !is_array($data['products'])) {
                dgm_proxy_error('no_products', 'هیچ محصولی برای این غرفه یافت نشد');
            }

            $created = 0;
            $skipped = 0;
            $items   = [];

            foreach ($data['products'] as $p) {
                $id_raw = $p['id'] ?? ($p['product_id'] ?? '');
                $name   = $p['name'] ?? ($p['title'] ?? '');
                $price  = isset($p['price']) ? (int)$p['price'] : 0;

                if (!$id_raw || !$name) {
                    $skipped++;
                    continue;
                }

                $id_norm = function_exists('basalam_normalize_id') ? basalam_normalize_id($id_raw) : (string)$id_raw;
                if ($id_norm === '0') {
                    $skipped++;
                    continue;
                }

                // ضد تکرار: اگر basalam_id قبلاً ثبت شده بود، رد کن
                $exists = 0;
                if (function_exists('basalam_product_exists_by_id')) {
                    $exists = (int) basalam_product_exists_by_id($id_norm);
                } else {
                    // جستجوی ساده متای basalam_id
                    global $wpdb;
                    $exists = (int)$wpdb->get_var(
                        $wpdb->prepare(
                            "SELECT post_id FROM {$wpdb->postmeta}
                             WHERE meta_key = %s AND meta_value = %s
                             ORDER BY post_id DESC LIMIT 1",
                            'basalam_id',
                            $id_norm
                        )
                    );
                }

                if ($exists > 0) {
                    $skipped++;
                    $items[] = [
                        'status'     => 'skipped',
                        'reason'     => 'duplicate',
                        'product_id' => $exists,
                        'basalam_id' => $id_norm,
                    ];
                    continue;
                }

                // ساخت محصول ساده
                $post_id = wp_insert_post([
                    'post_title'   => wp_strip_all_tags($name, true),
                    'post_type'    => 'product',
                    'post_status'  => $post_status,
                    'post_content' => '',
                    'post_excerpt' => '',
                ], true);

                if (is_wp_error($post_id) || !$post_id) {
                    $items[] = [
                        'status'     => 'error',
                        'reason'     => 'insert_failed',
                        'basalam_id' => $id_norm,
                    ];
                    continue;
                }

                if ($cat_id > 0) {
                    wp_set_object_terms($post_id, [$cat_id], 'product_cat');
                }

                if ($price > 0) {
                    update_post_meta($post_id, '_price', (string)$price);
                    update_post_meta($post_id, '_regular_price', (string)$price);
                }

                update_post_meta($post_id, 'basalam_id', $id_norm);
                update_post_meta($post_id, '_basalam_url', esc_url_raw($vendor_url));

                $created++;
                $items[] = [
                    'status'     => 'created',
                    'product_id' => (int)$post_id,
                    'basalam_id' => $id_norm,
                ];
            }

            dgm_proxy_json([
                'ok'      => true,
                'source'  => 'basalam',
                'created' => $created,
                'skipped' => $skipped,
                'items'   => $items,
            ]);
        }
        break;

        default:
            dgm_proxy_error('unknown_action', 'Unknown action: ' . $action);
    }

} catch (Throwable $e) {
    dgm_proxy_error('exception', $e->getMessage());
}
