<?php
require_once __DIR__ . '/config.php';
require_once __DIR__ . '/src/StateStore.php';
require_once __DIR__ . '/src/SettingsStore.php';
require_once __DIR__ . '/src/TelegramApi.php';

class DigiMasterBot {
    private $api;
    private $stateStore;
    private $settingsStore;

    public function __construct() {
        $this->api           = new TelegramApi(TG_BOT_TOKEN);
        $this->stateStore    = new StateStore(DM_STATE_FILE);
        $this->settingsStore = new SettingsStore(DM_SETTINGS_FILE);
    }

    /* ============== PROXY CALLER ============== */
    private function callProxy($action, array $payload) {
        $body = json_encode([
            'token'   => DGM_PROXY_TOKEN,
            'action'  => $action,
            'payload' => $payload,
        ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

        $ch = curl_init(DGM_PROXY_URL);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST           => true,
            CURLOPT_HTTPHEADER     => ['Content-Type: application/json'],
            CURLOPT_POSTFIELDS     => $body,
            CURLOPT_TIMEOUT        => 30,
        ]);
        $res = curl_exec($ch);
        curl_close($ch);

        if (!$res) return ['ok' => false, 'error' => ['code' => 'no_response', 'message' => 'No response from proxy']];
        $json = json_decode($res, true);
        if (!is_array($json)) {
            return ['ok' => false, 'error' => ['code' => 'bad_json', 'message' => 'Invalid JSON from proxy']];
        }
        return $json;
    }

    /* ============== MAIN DISPATCH ============== */
    public function handleUpdate(array $update) {
        if (!empty($update['message'])) {
            $this->handleMessage($update['message']);
        } elseif (!empty($update['callback_query'])) {
            $this->handleCallback($update['callback_query']);
        }
    }

    private function handleMessage(array $msg) {
        $chatId = $msg['chat']['id'];
        $text   = isset($msg['text']) ? trim($msg['text']) : '';

        $state    = $this->stateStore->get($chatId);
        $settings = $this->settingsStore->get($chatId);

        // اگر در حالت ویزارد تنظیمات هستیم
        if (!empty($state['mode']) && strpos($state['mode'], 'cfg:') === 0) {
            $this->handleConfigWizardMessage($chatId, $text, $state, $settings);
            return;
        }

        // اگر در حالت ویزارد ایمپورت هستیم
        if (!empty($state['mode']) && strpos($state['mode'], 'imp:') === 0) {
            $this->handleImportWizardMessage($chatId, $text, $state, $settings);
            return;
        }

        // فرمان‌های اصلی
        if ($text === '/start') {
            $this->showMainMenu($chatId);
            return;
        }

        // در حالت عادی، فقط منوی اصلی را نشان بده
        $this->showMainMenu($chatId);
    }

    private function handleCallback(array $cb) {
        $chatId = $cb['message']['chat']['id'];
        $data   = $cb['data'] ?? '';

        $state    = $this->stateStore->get($chatId);
        $settings = $this->settingsStore->get($chatId);

        // مثال data: "src:digikala" / "imp:digikala:single" / "cfg:torob:bulk"
        if (strpos($data, 'src:') === 0) {
            $source = substr($data, 4);
            $this->showSourceMenu($chatId, $source);
            return;
        }

        if (strpos($data, 'imp:') === 0) {
            $state['mode'] = $data; // مثلا imp:techno:bulk
            $state['step'] = 1;
            $state['buffer'] = [];
            $this->stateStore->set($chatId, $state);
            $this->askImportStep($chatId, $state, $settings);
            return;
        }

        if (strpos($data, 'cfg:') === 0) {
            $state['mode'] = $data; // مثلا cfg:techno:bulk
            $state['step'] = 1;
            $state['buffer'] = [];
            $this->stateStore->set($chatId, $state);
            $this->askConfigStep($chatId, $state, $settings);
            return;
        }
    }

    /* ============== MENUS ============== */
    private function showMainMenu($chatId) {
        $kb = [
            [
                ['text' => '🛒 دیجی‌کالا', 'callback_data' => 'src:digikala'],
                ['text' => '🧺 باسلام',    'callback_data' => 'src:basalam'],
            ],
            [
                ['text' => '🧭 ترب',       'callback_data' => 'src:torob'],
                ['text' => '🔌 تکنولایف',  'callback_data' => 'src:techno'],
            ],
        ];
        $this->api->sendMessage($chatId, "یکی از منابع را انتخاب کن:", [
            'inline_keyboard' => $kb,
        ]);
    }

    private function showSourceMenu($chatId, $source) {
        $rows = [];

        if ($source === 'digikala') {
            $rows[] = [
                ['text' => '🍏 ایمپورت تکی', 'callback_data' => 'imp:digikala:single'],
            ];
            // فعلاً فقط تنظیمات تکی را پیاده کردیم
            $rows[] = [
                ['text' => '⚙️ تنظیمات تکی', 'callback_data' => 'cfg:digikala:single'],
            ];
        } elseif ($source === 'basalam') {
            $rows[] = [
                ['text' => '🏪 ایمپورت غرفه (دسته‌ای)', 'callback_data' => 'imp:basalam:bulk'],
            ];
            $rows[] = [
                ['text' => '⚙️ تنظیمات غرفه', 'callback_data' => 'cfg:basalam:bulk'],
            ];
        } elseif ($source === 'torob') {
            $rows[] = [
                ['text' => '🧭 ایمپورت تکی',        'callback_data' => 'imp:torob:single'],
                ['text' => '🧭📦 ایمپورت دسته‌ای',   'callback_data' => 'imp:torob:bulk'],
            ];
            $rows[] = [
                ['text' => '⚙️ تنظیمات تکی',        'callback_data' => 'cfg:torob:single'],
                ['text' => '⚙️ تنظیمات دسته‌ای',     'callback_data' => 'cfg:torob:bulk'],
            ];
        } elseif ($source === 'techno') {
            $rows[] = [
                ['text' => '🔌 ایمپورت تکی',        'callback_data' => 'imp:techno:single'],
                ['text' => '🔌📦 ایمپورت دسته‌ای',   'callback_data' => 'imp:techno:bulk'],
            ];
            $rows[] = [
                ['text' => '⚙️ تنظیمات تکی',        'callback_data' => 'cfg:techno:single'],
                ['text' => '⚙️ تنظیمات دسته‌ای',     'callback_data' => 'cfg:techno:bulk'],
            ];
        }

        $this->api->sendMessage($chatId, "منوی {$source}:", [
            'inline_keyboard' => $rows,
        ]);
    }

    /* ============== CONFIG WIZARD ============== */

    private function configNeedsCategory($source, $sub) {
        // همه تنظیمات فعلی ما در گام اول دسته ووکامرس می‌خواهند
        return true;
    }

    private function ensureCategories($chatId, array &$state) {
        if (!empty($state['categories']) && is_array($state['categories'])) {
            return true;
        }
        $res = $this->callProxy('wc_categories', []);
        if (empty($res['ok']) || empty($res['categories']) || !is_array($res['categories'])) {
            $this->api->sendMessage($chatId, "❌ خطا در دریافت دسته‌بندی‌های ووکامرس. لطفاً بعداً دوباره تلاش کن یا شناسه دسته‌بندی را دستی وارد کن.");
            return false;
        }
        $state['categories'] = $res['categories'];
        $this->stateStore->set($chatId, $state);
        return true;
    }

    private function askCategorySelection($chatId, array $state) {
        if (empty($state['categories']) || !is_array($state['categories'])) {
            $this->api->sendMessage($chatId, "❌ دسته‌بندی‌ای یافت نشد. می‌توانی شناسهٔ دسته رو دستی وارد کنی.");
            return;
        }
        $lines = ["یکی از دسته‌بندی‌های زیر را انتخاب کن و *شماره* آن را بفرست:"];
        $i = 1;
        foreach ($state['categories'] as $cat) {
            $full = isset($cat['full_name']) ? $cat['full_name'] : $cat['name'];
            $lines[] = $i . ") " . $full . " (ID: " . $cat['id'] . ")";
            $i++;
            if ($i > 25) break; // برای جلوگیری از طولانی شدن پیام
        }
        $this->api->sendMessage($chatId, implode("\n", $lines));
    }

    private function askConfigStep($chatId, array $state, array $settings) {
        $mode = $state['mode'] ?? '';
        $step = (int)($state['step'] ?? 1);
        if (!$mode) return;

        // mode: cfg:source:sub
        $parts = explode(':', $mode);
        if (count($parts) !== 3) return;
        list(, $source, $sub) = $parts;

        $buffer = $state['buffer'] ?? [];
        $current = $settings[$source][$sub] ?? [];

        // گام ۱: انتخاب دسته ووکامرس
        if ($step === 1 && $this->configNeedsCategory($source, $sub)) {
            if ($this->ensureCategories($chatId, $state)) {
                $this->askCategorySelection($chatId, $state);
            }
            return;
        }

        // بعد از دسته، سایر تنظیمات بر اساس منبع
        switch ($source) {
            case 'digikala':
                if ($sub === 'single') {
                    if ($step === 2) {
                        $this->api->sendMessage($chatId, "وضعیت انتشار پیش‌فرض را انتخاب کن: `publish` یا `draft`");
                    } elseif ($step === 3) {
                        $this->api->sendMessage($chatId, "آیا تصاویر ایمپورت شوند؟ (بله/خیر)");
                    } elseif ($step === 4) {
                        $this->api->sendMessage($chatId, "آیا ویژگی‌ها و متغیرها ایمپورت شوند؟ (بله/خیر)");
                    } elseif ($step === 5) {
                        $this->api->sendMessage($chatId, "آیا توضیحات محصول ایمپورت شود؟ (بله/خیر)");
                    } elseif ($step === 6) {
                        $this->api->sendMessage($chatId, "حالت موجودی: فقط موجودها (`available`) یا همه (`all`)?");
                    }
                }
                break;

            case 'basalam':
                if ($sub === 'bulk') {
                    if ($step === 2) {
                        $this->api->sendMessage($chatId, "وضعیت پیش‌فرض انتشار محصولات غرفه: `publish` یا `draft`");
                    }
                }
                break;

            case 'torob':
                if ($sub === 'single') {
                    // فقط دسته نیاز دارد؛ پایان
                    $this->api->sendMessage($chatId, "تنظیمات ترب (ایمپورت تکی) فقط شامل دسته مقصد است. پس از انتخاب دسته، تنظیمات ذخیره می‌شود.");
                    // ذخیره در handleConfigWizardMessage انجام می‌شود
                } elseif ($sub === 'bulk') {
                    if ($step === 2) {
                        $this->api->sendMessage($chatId, "📄 تعداد صفحات جست‌جوی ترب را بنویس (مثلاً 1 تا 10):");
                    }
                }
                break;

            case 'techno':
                if ($sub === 'single') {
                    $this->api->sendMessage($chatId, "تنظیمات تکنولایف (ایمپورت تکی) فقط شامل دسته مقصد است. پس از انتخاب دسته، تنظیمات ذخیره می‌شود.");
                } elseif ($sub === 'bulk') {
                    if ($step === 2) {
                        $this->api->sendMessage($chatId, "📄 تعداد صفحات تکنولایف را بنویس (عدد 1 تا 10):");
                    } elseif ($step === 3) {
                        $this->api->sendMessage($chatId, "فقط کالاهای *موجود* ایمپورت شوند؟ (بله/خیر)");
                    }
                }
                break;
        }
    }

    private function handleConfigWizardMessage($chatId, $text, array $state, array $settings) {
        $mode = $state['mode'] ?? '';
        $step = (int)($state['step'] ?? 1);
        $buffer = $state['buffer'] ?? [];

        $parts = explode(':', $mode);
        if (count($parts) !== 3) return;
        list(, $source, $sub) = $parts;

        // STEP 1: دسته
        if ($step === 1 && $this->configNeedsCategory($source, $sub)) {
            // سعی کن از لیست دسته‌ها براساس شماره انتخاب کنی
            if (!empty($state['categories']) && is_array($state['categories']) && ctype_digit($text)) {
                $idx = (int)$text;
                if ($idx >= 1 && $idx <= count($state['categories'])) {
                    $cat = $state['categories'][$idx - 1];
                    $buffer['category'] = (int)$cat['id'];
                } else {
                    $this->api->sendMessage($chatId, "شماره نامعتبر است. دوباره یکی از شماره‌های فهرست را بفرست.");
                    return;
                }
            } elseif (ctype_digit($text)) {
                // کاربر مستقیم شناسه دسته را فرستاده
                $buffer['category'] = (int)$text;
            } else {
                $this->api->sendMessage($chatId, "لطفاً فقط شماره یا شناسهٔ دسته را به صورت عددی بفرست.");
                return;
            }

            $state['step'] = $step + 1;
            $state['buffer'] = $buffer;
            $this->stateStore->set($chatId, $state);

            $this->askConfigStep($chatId, $state, $settings);
            return;
        }

        // STEP >1: بر اساس منبع
        switch ($source) {
            case 'digikala':
                if ($sub === 'single') {
                    if ($step === 2) {
                        $val = strtolower($text) === 'publish' ? 'publish' : 'draft';
                        $buffer['post_status'] = $val;
                    } elseif ($step === 3) {
                        $buffer['import_images'] = (in_array(mb_strtolower($text), ['بله','بله.','yes','y'], true) ? 1 : 0);
                    } elseif ($step === 4) {
                        $buffer['import_attrs'] = (in_array(mb_strtolower($text), ['بله','yes','y'], true) ? 1 : 0);
                        $buffer['import_vars']  = $buffer['import_attrs'];
                    } elseif ($step === 5) {
                        $buffer['import_desc'] = (in_array(mb_strtolower($text), ['بله','yes','y'], true) ? 1 : 0);
                    } elseif ($step === 6) {
                        $v = strtolower($text);
                        $buffer['stock_mode'] = ($v === 'available') ? 'available' : 'all';
                        // پایان و ذخیره
                        $settings['digikala']['single'] = $buffer;
                        $this->settingsStore->set($chatId, $settings);
                        $this->stateStore->clear($chatId);
                        $this->api->sendMessage($chatId, "✅ تنظیمات دیجی‌کالا (ایمپورت تکی) ذخیره شد.");
                        return;
                    }
                }
                break;

            case 'basalam':
                if ($sub === 'bulk') {
                    if ($step === 2) {
                        $val = strtolower($text) === 'publish' ? 'publish' : 'draft';
                        $buffer['post_status'] = $val;

                        $settings['basalam']['bulk'] = $buffer;
                        $this->settingsStore->set($chatId, $settings);
                        $this->stateStore->clear($chatId);
                        $this->api->sendMessage($chatId, "✅ تنظیمات باسلام (ایمپورت غرفه) ذخیره شد.");
                        return;
                    }
                }
                break;

            case 'torob':
                if ($sub === 'single') {
                    // فقط دسته
                    $settings['torob']['single'] = $buffer;
                    $this->settingsStore->set($chatId, $settings);
                    $this->stateStore->clear($chatId);
                    $this->api->sendMessage($chatId, "✅ تنظیمات ترب (ایمپورت تکی) ذخیره شد.");
                    return;
                } elseif ($sub === 'bulk') {
                    if ($step === 2) {
                        if (!ctype_digit($text) || (int)$text < 1) {
                            $this->api->sendMessage($chatId, "لطفاً یک عدد بزرگ‌تر از صفر برای تعداد صفحات وارد کن.");
                            return;
                        }
                        $pages = (int)$text;
                        if ($pages > 20) $pages = 20;
                        $buffer['pages'] = $pages;

                        $settings['torob']['bulk'] = $buffer;
                        $this->settingsStore->set($chatId, $settings);
                        $this->stateStore->clear($chatId);
                        $this->api->sendMessage($chatId, "✅ تنظیمات ترب (ایمپورت دسته‌ای) ذخیره شد.");
                        return;
                    }
                }
                break;

            case 'techno':
                if ($sub === 'single') {
                    $settings['techno']['single'] = $buffer;
                    $this->settingsStore->set($chatId, $settings);
                    $this->stateStore->clear($chatId);
                    $this->api->sendMessage($chatId, "✅ تنظیمات تکنولایف (ایمپورت تکی) ذخیره شد.");
                    return;
                } elseif ($sub === 'bulk') {
                    if ($step === 2) {
                        if (!ctype_digit($text) || (int)$text < 1) {
                            $this->api->sendMessage($chatId, "لطفاً یک عدد بزرگ‌تر از صفر برای تعداد صفحات وارد کن.");
                            return;
                        }
                        $pages = (int)$text;
                        if ($pages > 20) $pages = 20;
                        $buffer['pages'] = $pages;

                        $state['step'] = 3;
                        $state['buffer'] = $buffer;
                        $this->stateStore->set($chatId, $state);
                        $this->askConfigStep($chatId, $state, $settings);
                        return;
                    } elseif ($step === 3) {
                        $buffer['stock_only'] = (in_array(mb_strtolower($text), ['بله','yes','y'], true) ? 1 : 0);

                        $settings['techno']['bulk'] = $buffer;
                        $this->settingsStore->set($chatId, $settings);
                        $this->stateStore->clear($chatId);
                        $this->api->sendMessage($chatId, "✅ تنظیمات تکنولایف (ایمپورت دسته‌ای) ذخیره شد.");
                        return;
                    }
                }
                break;
        }

        // اگر به اینجا رسیدیم یعنی هنوز مراحل ادامه دارد
        $state['step'] = $step + 1;
        $state['buffer'] = $buffer;
        $this->stateStore->set($chatId, $state);
        $this->askConfigStep($chatId, $state, $settings);
    }

    /* ============== IMPORT WIZARD ============== */

    private function askImportStep($chatId, array $state, array $settings) {
        $mode = $state['mode'] ?? '';
        $step = (int)($state['step'] ?? 1);

        $parts = explode(':', $mode);
        if (count($parts) !== 3) return;
        list(, $source, $sub) = $parts;

        switch ($source) {
            case 'digikala':
                if ($sub === 'single') {
                    if ($step === 1) {
                        $this->api->sendMessage($chatId, "لینک یا کد DKP دیجی‌کالا را بفرست:");
                    }
                }
                break;

            case 'basalam':
                if ($sub === 'bulk') {
                    if ($step === 1) {
                        $this->api->sendMessage($chatId, "لینک غرفه باسلام را بفرست (مثال: https://basalam.com/SHOP):");
                    }
                }
                break;

            case 'torob':
                if ($sub === 'single') {
                    if ($step === 1) {
                        $this->api->sendMessage($chatId, "یا PRK ترب را بفرست یا لینک محصول ترب را:");
                    }
                } elseif ($sub === 'bulk') {
                    if ($step === 1) {
                        $this->api->sendMessage($chatId, "لینک دسته‌بندی ترب را بفرست (مثال: https://torob.com/browse/?category=...):");
                    }
                }
                break;

            case 'techno':
                if ($sub === 'single') {
                    if ($step === 1) {
                        $this->api->sendMessage($chatId, "لینک محصول تکنولایف را بفرست:");
                    }
                } elseif ($sub === 'bulk') {
                    if ($step === 1) {
                        $this->api->sendMessage($chatId, "لینک صفحه دسته/جستجوی تکنولایف را بفرست (مثال: صفحه موبایل‌ها):");
                    }
                }
                break;
        }
    }

    private function handleImportWizardMessage($chatId, $text, array $state, array $settings) {
        $mode = $state['mode'] ?? '';
        $step = (int)($state['step'] ?? 1);
        $buffer = $state['buffer'] ?? [];

        $parts = explode(':', $mode);
        if (count($parts) !== 3) return;
        list(, $source, $sub) = $parts;

        switch ($source) {
            case 'digikala':
                if ($sub === 'single' && $step === 1) {
                    $buffer['id_or_url'] = $text;

                    $opts = $settings['digikala']['single'] ?? [];
                    $payload = [
                        'id_or_url' => $buffer['id_or_url'],
                        'opts'      => $opts,
                    ];

                    $this->api->sendMessage($chatId, "⏳ در حال ایمپورت دیجی‌کالا...");
                    $res = $this->callProxy('digikala_single', $payload);
                    if (!empty($res['ok']) && !empty($res['product_id'])) {
                        $this->api->sendMessage($chatId, "✅ ایمپورت موفق بود.\nID محصول ووکامرس: `".$res['product_id']."`");
                    } else {
                        $err = $res['error']['message'] ?? 'خطای نامشخص';
                        $code = $res['error']['code'] ?? 'import_failed';
                        $this->api->sendMessage($chatId, "❌ خطا در ایمپورت.\nError: *{$code}*\n{$err}");
                    }

                    $this->stateStore->clear($chatId);
                    return;
                }
                break;

            case 'basalam':
                if ($sub === 'bulk' && $step === 1) {
                    $buffer['vendor_url'] = $text;

                    $opts = $settings['basalam']['bulk'] ?? [];
                    $payload = [
                        'vendor_url'  => $buffer['vendor_url'],
                        'category'    => (int)($opts['category'] ?? 0),
                        'post_status' => $opts['post_status'] ?? 'draft',
                    ];

                    $this->api->sendMessage($chatId, "⏳ در حال ایمپورت غرفه باسلام...");
                    $res = $this->callProxy('basalam_bulk', $payload);
                    if (!empty($res['ok'])) {
                        $this->api->sendMessage(
                            $chatId,
                            "✅ ایمپورت غرفه باسلام تمام شد.\nمحصولات جدید: `{$res['created']}`\nتکراری/رد شده: `{$res['skipped']}`"
                        );
                    } else {
                        $err = $res['error']['message'] ?? 'خطای نامشخص';
                        $code = $res['error']['code'] ?? 'import_failed';
                        $this->api->sendMessage($chatId, "❌ خطا در ایمپورت غرفه.\nError: *{$code}*\n{$err}");
                    }

                    $this->stateStore->clear($chatId);
                    return;
                }
                break;

            case 'torob':
                if ($sub === 'single' && $step === 1) {
                    $opts = $settings['torob']['single'] ?? [];
                    $payload = [
                        'category' => (int)($opts['category'] ?? 0),
                    ];

                    if (stripos($text, 'http') === 0) {
                        $payload['url'] = $text;
                    } else {
                        $payload['prk'] = $text;
                    }

                    $this->api->sendMessage($chatId, "⏳ در حال ایمپورت تکی ترب...");
                    $res = $this->callProxy('torob_single', $payload);
                    if (!empty($res['ok']) && !empty($res['product_id'])) {
                        $this->api->sendMessage($chatId, "✅ ایمپورت ترب موفق بود.\nID محصول ووکامرس: `".$res['product_id']."`");
                    } else {
                        $err = $res['error']['message'] ?? 'خطای نامشخص';
                        $code = $res['error']['code'] ?? 'import_failed';
                        $this->api->sendMessage($chatId, "❌ خطا در ایمپورت ترب.\nError: *{$code}*\n{$err}");
                    }

                    $this->stateStore->clear($chatId);
                    return;
                } elseif ($sub === 'bulk' && $step === 1) {
                    $buffer['category_url'] = $text;

                    $opts = $settings['torob']['bulk'] ?? [];
                    $payload = [
                        'category_url' => $buffer['category_url'],
                        'category'     => (int)($opts['category'] ?? 0),
                        'pages'        => (int)($opts['pages'] ?? 1),
                    ];

                    $this->api->sendMessage($chatId, "⏳ در حال ایمپورت دسته‌ای ترب...");
                    $res = $this->callProxy('torob_bulk', $payload);
                    if (!empty($res['ok'])) {
                        $this->api->sendMessage(
                            $chatId,
                            "✅ ایمپورت دسته‌ای ترب تمام شد.\nPRKهای پیدا شده: `{$res['total_prks']}`\nجدید: `{$res['created']}`\nبروزرسانی‌شده: `{$res['updated']}`\nخطا: `{$res['errors']}`"
                        );
                    } else {
                        $err = $res['error']['message'] ?? 'خطای نامشخص';
                        $code = $res['error']['code'] ?? 'import_failed';
                        $this->api->sendMessage($chatId, "❌ خطا در ایمپورت دسته‌ای ترب.\nError: *{$code}*\n{$err}");
                    }

                    $this->stateStore->clear($chatId);
                    return;
                }
                break;

            case 'techno':
                if ($sub === 'single' && $step === 1) {
                    $buffer['url'] = $text;

                    $opts = $settings['techno']['single'] ?? [];
                    $payload = [
                        'url'      => $buffer['url'],
                        'category' => (int)($opts['category'] ?? 0),
                    ];

                    $this->api->sendMessage($chatId, "⏳ در حال ایمپورت تکی تکنولایف...");
                    $res = $this->callProxy('techno_single', $payload);
                    if (!empty($res['ok']) && !empty($res['product_id'])) {
                        $this->api->sendMessage($chatId, "✅ ایمپورت تکنولایف موفق بود.\nID محصول ووکامرس: `".$res['product_id']."`");
                    } else {
                        $err = $res['error']['message'] ?? 'خطای نامشخص';
                        $code = $res['error']['code'] ?? 'import_failed';
                        $this->api->sendMessage($chatId, "❌ خطا در ایمپورت تکنولایف.\nError: *{$code}*\n{$err}");
                    }

                    $this->stateStore->clear($chatId);
                    return;
                } elseif ($sub === 'bulk' && $step === 1) {
                    $buffer['url'] = $text;

                    $opts = $settings['techno']['bulk'] ?? [];
                    $payload = [
                        'url'        => $buffer['url'],
                        'category'   => (int)($opts['category'] ?? 0),
                        'pages'      => (int)($opts['pages'] ?? 1),
                        'stock_only' => (int)($opts['stock_only'] ?? 0),
                    ];

                    $this->api->sendMessage($chatId, "⏳ در حال ایمپورت دسته‌ای تکنولایف...");
                    $res = $this->callProxy('techno_bulk', $payload);
                    if (!empty($res['ok'])) {
                        $this->api->sendMessage(
                            $chatId,
                            "✅ ایمپورت دسته‌ای تکنولایف تمام شد.\nلینک‌ها: `{$res['total_links']}`\nجدید: `{$res['created']}`\nتکراری: `{$res['skipped']}`\nخطا: `{$res['errors']}`"
                        );
                    } else {
                        $err = $res['error']['message'] ?? 'خطای نامشخص';
                        $code = $res['error']['code'] ?? 'import_failed';
                        $this->api->sendMessage($chatId, "❌ خطا در ایمپورت دسته‌ای تکنولایف.\nError: *{$code}*\n{$err}");
                    }

                    $this->stateStore->clear($chatId);
                    return;
                }
                break;
        }

        // اگر به اینجا رسیدیم، ویزارد نامعتبر است؛ ریست شود
        $this->stateStore->clear($chatId);
        $this->api->sendMessage($chatId, "ویزارد ایمپورت پایان یافت یا نامعتبر بود. دوباره از منو شروع کن.");
    }
}
