<?php
/**
 * Star-Phone Product Details Web Service (PHP 7 compatible)
 * Usage:
 *   star_phone_single.php?url=https://star-phone.ir/product/p-15804/... [&debug=1]
 */

header('Content-Type: application/json; charset=utf-8');

/* -------- debug switch -------- */
$debug = isset($_GET['debug']) && $_GET['debug'] == '1';
if ($debug) {
  error_reporting(E_ALL);
  ini_set('display_errors', '1');
} else {
  error_reporting(E_ALL);
  ini_set('display_errors', '0');
}

/* -------- polyfills -------- */
if (!function_exists('str_starts_with')) {
  function str_starts_with($haystack, $needle) {
    return $needle !== '' && substr($haystack, 0, strlen($needle)) === $needle;
  }
}

/* -------- helpers -------- */
function http_json($status, $data){
  http_response_code($status);
  echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
  exit;
}

function http_get($url, $timeout = 20) {
  $ch = curl_init();
  curl_setopt_array($ch, [
    CURLOPT_URL            => $url,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_MAXREDIRS      => 5,
    CURLOPT_CONNECTTIMEOUT => 10,
    CURLOPT_TIMEOUT        => $timeout,
    CURLOPT_SSL_VERIFYPEER => true,
    CURLOPT_SSL_VERIFYHOST => 2,
    CURLOPT_USERAGENT      => 'Mozilla/5.0 (compatible; StarPhoneSingle/1.1)',
    CURLOPT_HTTPHEADER     => [
      'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
      'Accept-Language: fa-IR,fa;q=0.9,en;q=0.8',
    ],
    CURLOPT_ENCODING       => ''
  ]);
  $body = curl_exec($ch);
  $err  = curl_error($ch);
  $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);
  return [$code, $body, $err];
}

function http_post_json($url, $data, $timeout = 20) {
  $payload = json_encode($data, JSON_UNESCAPED_UNICODE);
  $ch = curl_init();
  curl_setopt_array($ch, [
    CURLOPT_URL            => $url,
    CURLOPT_POST           => true,
    CURLOPT_POSTFIELDS     => $payload,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_MAXREDIRS      => 5,
    CURLOPT_CONNECTTIMEOUT => 10,
    CURLOPT_TIMEOUT        => $timeout,
    CURLOPT_SSL_VERIFYPEER => true,
    CURLOPT_SSL_VERIFYHOST => 2,
    CURLOPT_USERAGENT      => 'Mozilla/5.0 (compatible; StarPhoneSingle/1.1)',
    CURLOPT_HTTPHEADER     => [
      'Content-Type: application/json',
      'Accept: application/json',
      'Accept-Language: fa-IR,fa;q=0.9,en;q=0.8',
    ],
    CURLOPT_ENCODING       => ''
  ]);
  $body = curl_exec($ch);
  $err  = curl_error($ch);
  $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);
  return [$code, $body, $err];
}

function normalize_url($url) {
  $parts = @parse_url($url);
  if (!$parts || empty($parts['host'])) return $url;
  $scheme = isset($parts['scheme']) ? $parts['scheme'] : 'https';
  $host   = $parts['host'];
  $path   = '';
  if (isset($parts['path'])) {
    $segments = explode('/', $parts['path']);
    foreach ($segments as &$seg) {
      if ($seg === '' || $seg === '.') continue;
      $seg = rawurlencode(rawurldecode($seg));
    }
    $path = implode('/', $segments);
  }
  $query = isset($parts['query']) ? ('?'.$parts['query']) : '';
  $frag  = isset($parts['fragment']) ? ('#'.$parts['fragment']) : '';
  return $scheme.'://'.$host.$path.$query.$frag;
}

function new_dom_xp($html) {
  libxml_use_internal_errors(true);
  $dom = new DOMDocument();
  @$dom->loadHTML($html);
  libxml_clear_errors();
  $xp = new DOMXPath($dom);
  return [$dom, $xp];
}

function text($node) {
  return trim(preg_replace('/\s+/u', ' ', html_entity_decode($node ? $node->textContent : '', ENT_QUOTES | ENT_HTML5, 'UTF-8')));
}
function to_int_price($str) {
  $s = str_replace([",", "٫", "٬", " "], "", $str);
  $s = preg_replace('/[^\d]/', '', $s);
  return $s === '' ? null : (int)$s;
}
function abs_url($base, $rel) {
  if (!$rel) return null;
  if (preg_match('~^https?://~i', $rel)) return $rel;
  $p = parse_url($base);
  $scheme = isset($p['scheme']) ? $p['scheme'] : 'https';
  $host   = isset($p['host']) ? $p['host'] : '';
  if (str_starts_with($rel, '//')) return $scheme.':'.$rel;
  if (str_starts_with($rel, '/'))  return $scheme.'://'.$host.$rel;
  $dir = isset($p['path']) ? preg_replace('~/[^/]*$~', '/', $p['path']) : '/';
  return $scheme.'://'.$host.$dir.$rel;
}
function take_unique(array $arr) {
  $arr = array_map('trim', $arr);
  $arr = array_filter($arr, function($x){ return $x !== ''; });
  return array_values(array_unique($arr));
}

/* -------- input -------- */
$url = isset($_GET['url']) ? trim($_GET['url']) : '';
if ($url === '') http_json(400, ['error'=>'Missing url parameter']);

$norm = normalize_url($url);
list($code, $html, $err) = http_get($norm);
if ($code < 200 || $code >= 400 || !$html) {
  http_json(502, ['source'=>$norm, 'error'=>'FetchFailed', 'http_code'=>$code, 'curl_error'=>$err]);
}

list($dom, $xp) = new_dom_xp($html);

/* -------- extract basics -------- */
$out = [
  'url'          => $norm,
  'id'           => null,
  'slug'         => null,
  'name'         => null,
  'price'        => null,   // تومان
  'currency'     => 'TOMAN',
  'images'       => [],
  'description'  => '',
  'brand'        => null,
  'category'     => [],
  'availability' => null,
  'rating'       => null,
  'models'       => [],
  'debug'        => []
];

$pu = @parse_url($norm);
$out['slug'] = isset($pu['path']) ? trim($pu['path'], '/') : '';

$nameNodes = $xp->query("//h1 | //meta[@property='og:title']/@content");
if ($nameNodes && $nameNodes->length) $out['name'] = text($nameNodes->item(0));

$priceCandidates = [];
foreach ([
  "//*[@id='finalPrice']",
  "//*[@id='price' and contains(@class,'product-price')]",
  "//*[contains(@class,'product-price-sale')]//span[contains(@class,'ng-binding')]",
  "//meta[@property='product:price:amount']/@content"
] as $q) {
  $n = $xp->query($q);
  if ($n && $n->length) $priceCandidates[] = text($n->item(0));
}
foreach ($priceCandidates as $pc) {
  $val = to_int_price($pc);
  if ($val) { $out['price'] = $val; break; }
}

$imgs = [];
foreach ($xp->query("//img[@id='mainImage']") as $n) {
  $src = $n->getAttribute('src') ?: $n->getAttribute('srcset');
  if (!$src) $src = $n->getAttribute('bsrc');
  if ($src) $imgs[] = abs_url($norm, $src);
}
foreach ($xp->query("//img[@bsrc]") as $n) {
  $imgs[] = abs_url($norm, $n->getAttribute('bsrc'));
}
foreach ($xp->query("//div[contains(@class,'gallery')]//img[@src]") as $n) {
  $imgs[] = abs_url($norm, $n->getAttribute('src'));
}
$out['images'] = take_unique($imgs);

$desc = '';
foreach ([
  "//*[@id='productDescription']",
  "//*[contains(@class,'product-description')]",
  "//meta[@property='og:description']/@content"
] as $q) {
  $n = $xp->query($q);
  if ($n && $n->length) { $desc = text($n->item(0)); if ($desc) break; }
}
$out['description'] = $desc;

$brandNode = $xp->query("//*[contains(@class,'brand')]/a | //*[contains(@class,'product-brand')]/a");
if ($brandNode && $brandNode->length) $out['brand'] = text($brandNode->item(0));

$crumbs = [];
foreach ($xp->query("//*[contains(@class,'breadcrumb')]//a") as $a) $crumbs[] = text($a);
$out['category'] = take_unique($crumbs);

$idNode = $xp->query("//*[@id='productInfo']");
if ($idNode && $idNode->length) {
  $out['id'] = $idNode->item(0)->getAttribute('data-product-id') ?: null;
}
$isMobo = false;
$moboNode = $xp->query("//*[@id='productMobo']");
if ($moboNode && $moboNode->length) {
  $val = strtolower(trim($moboNode->item(0)->getAttribute('data-is-mobo')));
  $isMobo = in_array($val, ['true','1','yes'], true);
}

/* -------- models from DOM (if rendered) -------- */
$models = [];
foreach ($xp->query("//div[contains(@class,'variety')][@data-variety-id]") as $v) {
  $name = '';
  foreach ($v->childNodes as $c) {
    if ($c instanceof DOMElement && strpos($c->getAttribute('class'), 'model-name') !== false) {
      $name = text($c); break;
    }
  }
  $mid   = $v->getAttribute('data-variety-id') ?: null;
  $p     = to_int_price($v->getAttribute('data-variety-price') ?: '');
  $pf    = to_int_price($v->getAttribute('data-variety-final-price') ?: '');
  $disc  = to_int_price($v->getAttribute('data-variety-discount') ?: '');
  $stock = to_int_price($v->getAttribute('stock') ?: '');

  if ($name || $mid) {
    $models[] = [
      'id'          => $mid,
      'name'        => $name,
      'price'       => $p,
      'final_price' => $pf ?: $p,
      'discount'    => $disc,
      'stock'       => $stock
    ];
  }
}

/* -------- models from internal API (if not rendered & isMobo) -------- */
if (empty($models) && $out['id'] && $isMobo) {
  $apiBase = sprintf('%s://%s', isset($pu['scheme']) ? $pu['scheme'] : 'https', isset($pu['host']) ? $pu['host'] : 'star-phone.ir');
  $api = rtrim($apiBase, '/').'/varieties/update';
  list($sc, $body, $e2) = http_post_json($api, ['productId' => (int)$out['id']]);
  if ($sc >= 200 && $sc < 400 && $body) {
    $j = json_decode($body, true);
    if (is_array($j)) {
      foreach ($j as $row) {
        $models[] = [
          'id'          => isset($row['id']) ? $row['id'] : null,
          'name'        => isset($row['text']) ? $row['text'] : null,
          'price'       => isset($row['formattedPrice']) ? to_int_price($row['formattedPrice']) : null,
          'final_price' => isset($row['formatedFinalPrice']) ? to_int_price($row['formatedFinalPrice']) : (isset($row['formattedPrice']) ? to_int_price($row['formattedPrice']) : null),
          'discount'    => isset($row['discount']) ? (int)$row['discount'] : null,
          'stock'       => isset($row['stock']) ? (int)$row['stock'] : null,
        ];
      }
    }
  } else {
    // برای بررسی راحت‌تر هنگام دیباگ
    if ($GLOBALS['debug']) {
      $out['debug'][] = ['api_status'=>$sc, 'api_error'=>$e2, 'api_body'=>$body];
    }
  }
}
$out['models'] = $models;

// اگر price خالی بود از اولین واریشن پر کن
if (!$out['price'] && !empty($models)) {
  $out['price'] = $models[0]['final_price'] ? $models[0]['final_price'] : $models[0]['price'];
}

// موجودی کلی
$stockText = $xp->query("//*[contains(@id,'noStockText') or contains(@class,'no-stock')]");
if ($stockText && $stockText->length) $out['availability'] = 'OUT_OF_STOCK';

// rating از JSON-LD اگر وجود داشت
foreach ($xp->query("//script[@type='application/ld+json']") as $s) {
  $j = json_decode($s->textContent, true);
  if (is_array($j)) {
    $arr = isset($j['@type']) ? [$j] : (is_array($j) ? $j : []);
    foreach ($arr as $it) {
      if (isset($it['@type']) && strtolower($it['@type']) === 'product') {
        if (!$out['name'] && !empty($it['name'])) $out['name'] = trim($it['name']);
        if (isset($it['image'])) {
          $imgs2 = is_array($it['image']) ? $it['image'] : [$it['image']];
          foreach ($imgs2 as $im) $out['images'][] = abs_url($norm, $im);
        }
        if (isset($it['offers']['price']) && !$out['price']) {
          $out['price'] = to_int_price($it['offers']['price']);
        }
        if (isset($it['aggregateRating']['ratingValue'])) {
          $out['rating'] = [
            'value' => (string)$it['aggregateRating']['ratingValue'],
            'count' => isset($it['aggregateRating']['reviewCount']) ? (string)$it['aggregateRating']['reviewCount'] : null,
          ];
        }
      }
    }
  }
}
$out['images'] = take_unique($out['images']);

http_json(200, $out);