<?php
/**
 * Simple WooCommerce Product Info API
 * GET param: url=https://example.com/product/slug
 * Returns: JSON with product fields
 */

header('Content-Type: application/json; charset=utf-8');

// ---------- helpers ----------
function respond($arr, $code = 200) {
    http_response_code($code);
    echo json_encode($arr, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

function is_private_ip($host) {
    // Prevent SSRF: block local/loopback/link-local/private ranges
    $ips = gethostbynamel($host);
    if (!$ips) return false; // couldn't resolve; let cURL handle
    foreach ($ips as $ip) {
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) === false) {
            return true;
        }
    }
    return false;
}

function fetch($url) {
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_MAXREDIRS => 5,
        CURLOPT_CONNECTTIMEOUT => 8,
        CURLOPT_TIMEOUT => 20,
        CURLOPT_USERAGENT => 'Mozilla/5.0 (compatible; WCProductFetcher/1.0)',
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_HTTPHEADER => ['Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8']
    ]);
    $body = curl_exec($ch);
    $err  = curl_error($ch);
    $info = curl_getinfo($ch);
    curl_close($ch);
    if ($body === false) {
        respond(["error" => "خطا در دریافت صفحه", "detail" => $err], 502);
    }
    // rudimentary size cap
    if (strlen($body) > 8 * 1024 * 1024) {
        respond(["error" => "حجم صفحه بیش از حد مجاز است."], 413);
    }
    return [$body, $info];
}

function load_dom($html) {
    libxml_use_internal_errors(true);
    $dom = new DOMDocument();
    // Handle encoding
    if (stripos($html, '<meta charset=') === false && stripos($html, 'charset=') === false) {
        $html = "<meta http-equiv='Content-Type' content='text/html; charset=utf-8'>".$html;
    }
    $dom->loadHTML($html);
    libxml_clear_errors();
    return $dom;
}

function xpath_query($xp, $query) {
    $nodes = @$xp->query($query);
    return $nodes ?: [];
}

function textContent($node) {
    return trim(preg_replace('/\s+/', ' ', $node->textContent ?? ''));
}

// ---------- validate input ----------
if (empty($_GET['url'])) respond(["error" => "پارامتر url الزامی است."], 400);
$url = trim($_GET['url']);

if (!filter_var($url, FILTER_VALIDATE_URL)) respond(["error" => "آدرس معتبر نیست."], 400);

// only http/https
$scheme = parse_url($url, PHP_URL_SCHEME);
if (!in_array(strtolower($scheme), ['http','https'])) respond(["error"=>"فقط http/https مجاز است."], 400);

// prevent SSRF to private IPs
$host = parse_url($url, PHP_URL_HOST);
if ($host && is_private_ip($host)) respond(["error" => "دسترسی به آدرس‌های داخلی مجاز نیست."], 403);

// ---------- fetch ----------
list($html, $info) = fetch($url);
$dom = load_dom($html);
$xp  = new DOMXPath($dom);

// ---------- extract via JSON-LD (preferred) ----------
$jsonldAll = [];
$product = [
    "name" => null,
    "sku" => null,
    "brand" => null,
    "description" => null,
    "short_description" => null,
    "price" => null,
    "regular_price" => null,
    "sale_price" => null,
    "currency" => null,
    "availability" => null,
    "stock_status" => null,
    "stock_quantity" => null,
    "rating" => null,
    "review_count" => null,
    "images" => [],
    "categories" => [],
    "attributes" => [],
    "variations" => []
];

foreach (xpath_query($xp, "//script[@type='application/ld+json']") as $s) {
    $json = trim($s->textContent);
    // Some themes put multiple JSON objects or invalid trailing commas; try to normalize:
    $candidates = [];
    if (preg_match_all('/\{.*\}/sU', $json, $m)) $candidates = $m[0];
    if (!$candidates) $candidates = [$json];

    foreach ($candidates as $cand) {
        $data = json_decode($cand, true);
        if (!$data) continue;
        $list = isset($data['@graph']) && is_array($data['@graph']) ? $data['@graph'] : (isset($data[0]) ? $data : [$data]);
        foreach ($list as $node) {
            if (!is_array($node)) continue;
            $type = $node['@type'] ?? null;
            if (is_array($type) && in_array('Product', $type) || $type === 'Product') {
                $jsonldAll[] = $node;
            }
        }
    }
}

if ($jsonldAll) {
    // take first product node
    $p = $jsonldAll[0];
    $product["name"] = $p["name"] ?? $product["name"];
    $product["sku"] = $p["sku"] ?? $product["sku"];
    $product["brand"] = is_array($p["brand"] ?? null) ? ($p["brand"]["name"] ?? null) : ($p["brand"] ?? null);
    $product["description"] = $p["description"] ?? $product["description"];
    if (!empty($p["image"])) {
        $imgs = is_array($p["image"]) ? $p["image"] : [$p["image"]];
        $product["images"] = array_values(array_unique(array_filter($imgs)));
    }
    // Offers
    $offers = $p["offers"] ?? null;
    if ($offers) {
        $offer = is_array($offers) && isset($offers[0]) ? $offers[0] : $offers;
        $product["price"] = $offer["price"] ?? $product["price"];
        $product["currency"] = $offer["priceCurrency"] ?? $product["currency"];
        $product["availability"] = $offer["availability"] ?? $product["availability"];
        if (isset($offer["availability"])) {
            $product["stock_status"] = (stripos($offer["availability"], 'InStock') !== false) ? "instock" :
                                       ((stripos($offer["availability"], 'OutOfStock') !== false) ? "outofstock" : null);
        }
    }
    // AggregateRating
    if (!empty($p["aggregateRating"])) {
        $ar = $p["aggregateRating"];
        $product["rating"] = $ar["ratingValue"] ?? null;
        $product["review_count"] = $ar["reviewCount"] ?? $ar["ratingCount"] ?? null;
    }
}

// ---------- extract via meta tags (Open Graph / product) ----------
$metas = [
    "og:title" => null,
    "og:description" => null,
    "og:image" => [],
    "product:price:amount" => null,
    "product:price:currency" => null,
    "product:availability" => null
];
foreach (xpath_query($xp, "//meta[@property or @name]") as $m) {
    $prop = $m->getAttribute('property') ?: $m->getAttribute('name');
    $content = trim($m->getAttribute('content'));
    if (!$prop || $content === '') continue;
    if ($prop === 'og:image') $metas["og:image"][] = $content;
    if (array_key_exists($prop, $metas) && $prop !== 'og:image') $metas[$prop] = $content;
}

if (!$product["name"] && $metas["og:title"]) $product["name"] = $metas["og:title"];
if (!$product["description"] && $metas["og:description"]) $product["description"] = $metas["og:description"];
if (!$product["price"] && $metas["product:price:amount"]) $product["price"] = $metas["product:price:amount"];
if (!$product["currency"] && $metas["product:price:currency"]) $product["currency"] = $metas["product:price:currency"];
if (!$product["availability"] && $metas["product:availability"]) $product["availability"] = $metas["product:availability"];
if (empty($product["images"]) && !empty($metas["og:image"])) $product["images"] = array_values(array_unique($metas["og:image"]));

// ---------- extract via WooCommerce HTML selectors ----------
$titleNode = xpath_query($xp, "//h1[contains(@class,'product_title') or contains(@class,'entry-title')]")[0] ?? null;
if (!$product["name"] && $titleNode) $product["name"] = textContent($titleNode);

// price: try common Woo selectors
$priceNode = xpath_query($xp, "//*[contains(@class,'price')]//*[contains(@class,'amount')]|//*[contains(@class,'price') and @data-price]");
if (!$product["price"] && $priceNode && $priceNode->length) {
    $priceText = preg_replace('/[^\d\.,]/', '', textContent($priceNode[0]));
    $product["price"] = str_replace(',', '.', preg_replace('/\.(?=.*\.)/', '', $priceText)); // normalize
}
// currency from symbol near price (best-effort)
if (!$product["currency"]) {
    $currencyMeta = xpath_query($xp, "//meta[@itemprop='priceCurrency']")[0] ?? null;
    if ($currencyMeta) $product["currency"] = $currencyMeta->getAttribute('content');
}

// regular/sale prices
$reg = xpath_query($xp, "//*[contains(@class,'price')]//*[contains(@class,'woocommerce-Price-amount') and ancestor::*[contains(@class,'price') and contains(@class,'del')]]");
$sale = xpath_query($xp, "//*[contains(@class,'price')]//*[contains(@class,'woocommerce-Price-amount') and ancestor::*[contains(@class,'price') and contains(@class,'ins')]]");
if ($reg->length) $product["regular_price"] = preg_replace('/[^\d\.,]/', '', textContent($reg[0]));
if ($sale->length) $product["sale_price"]    = preg_replace('/[^\d\.,]/', '', textContent($sale[0]));

// SKU
$skuNode = xpath_query($xp, "//*[contains(@class,'sku') or @itemprop='sku']")[0] ?? null;
if ($skuNode) $product["sku"] = textContent($skuNode);

// stock
$stockNode = xpath_query($xp, "//*[contains(@class,'stock')]")[0] ?? null;
if ($stockNode && !$product["stock_status"]) {
    $txt = mb_strtolower(textContent($stockNode));
    if (strpos($txt,'ناموجود') !== false || strpos($txt,'out') !== false) $product["stock_status"] = "outofstock";
    elseif (strpos($txt,'موجود') !== false || strpos($txt,'in stock') !== false) $product["stock_status"] = "instock";
}

// rating + reviews
$ratingNode = xpath_query($xp, "//*[@itemprop='ratingValue' or contains(@class,'star-rating')]")[0] ?? null;
if ($ratingNode && !$product["rating"]) {
    $product["rating"] = $ratingNode->getAttribute('content') ?: preg_replace('/[^\d\.,]/','', textContent($ratingNode));
}
$revNode = xpath_query($xp, "//*[@itemprop='reviewCount' or contains(@class,'count')]")[0] ?? null;
if ($revNode && !$product["review_count"]) {
    $product["review_count"] = intval(preg_replace('/\D/','', textContent($revNode)));
}

// breadcrumb categories
foreach (xpath_query($xp, "//*[contains(@class,'woocommerce-breadcrumb')]//a") as $a) {
    $txt = textContent($a);
    if ($txt) $product["categories"][] = $txt;
}
$product["categories"] = array_values(array_unique($product["categories"]));

// images from gallery
$imgUrls = [];
foreach (xpath_query($xp, "//*[contains(@class,'woocommerce-product-gallery')]//img") as $img) {
    $srcs = [
        $img->getAttribute('data-large_image'),
        $img->getAttribute('data-src'),
        $img->getAttribute('data-o_src'),
        $img->getAttribute('src')
    ];
    foreach ($srcs as $s) if ($s) $imgUrls[] = $s;
}
if ($imgUrls) $product["images"] = array_values(array_unique(array_merge($product["images"], $imgUrls)));

// attributes table
$attrs = [];
foreach (xpath_query($xp, "//*[contains(@class,'woocommerce-product-attributes') or contains(@class,'shop_attributes')]//tr") as $tr) {
    $labelNode = xpath_query($xp, ".//th|.//*[contains(@class,'label')]", $tr)[0] ?? null;
    $valueNode = xpath_query($xp, ".//td|.//*[contains(@class,'value')]", $tr)[0] ?? null;
}
// DOMXPath doesn't support context params in this style; do it manually:
$tables = xpath_query($xp, "//*[contains(@class,'woocommerce-product-attributes') or contains(@class,'shop_attributes')]");
foreach ($tables as $table) {
    foreach ($table->getElementsByTagName('tr') as $tr) {
        $label = null; $value = null;
        foreach ($tr->getElementsByTagName('th') as $th) $label = textContent($th);
        foreach ($tr->getElementsByTagName('td') as $td) $value = textContent($td);
        if ($label && $value) $attrs[$label] = $value;
    }
}
$product["attributes"] = $attrs;

// short/long description
$shortDesc = xpath_query($xp, "//*[contains(@class,'woocommerce-product-details__short-description')]");
if ($shortDesc && $shortDesc->length) $product["short_description"] = textContent($shortDesc[0]);
$desc = xpath_query($xp, "//*[@id='tab-description' or contains(@class,'woocommerce-Tabs-panel--description') or contains(@class,'product-desc')]");
if ($desc && $desc->length) $product["description"] = $product["description"] ?: textContent($desc[0]);

// variations (if variable product)
$varForm = xpath_query($xp, "//form[contains(@class,'variations_form')]");
if ($varForm && $varForm->length) {
    $form = $varForm[0];
    $json = $form->getAttribute('data-product_variations');
    if ($json) {
        $vars = json_decode(html_entity_decode($json, ENT_QUOTES | ENT_HTML5), true);
        if (is_array($vars)) {
            $clean = [];
            foreach ($vars as $v) {
                $clean[] = [
                    "variation_id"   => $v["variation_id"] ?? null,
                    "sku"            => $v["sku"] ?? null,
                    "price_html"     => $v["price_html"] ?? null,
                    "display_price"  => $v["display_price"] ?? null,
                    "display_regular_price" => $v["display_regular_price"] ?? null,
                    "is_in_stock"    => $v["is_in_stock"] ?? null,
                    "attributes"     => $v["attributes"] ?? []
                ];
            }
            $product["variations"] = $clean;
        }
    }
}

// final fallback: page title
if (!$product["name"]) {
    $t = xpath_query($xp, "//title")[0] ?? null;
    if ($t) $product["name"] = textContent($t);
}

// normalize numbers (replace , with . if needed)
foreach (["price","regular_price","sale_price","rating"] as $k) {
    if ($product[$k]) {
        $val = str_replace(['٫', ','], ['.', '.'], $product[$k]); // Persian comma etc.
        $val = preg_replace('/[^\d\.]/','',$val);
        $product[$k] = $val !== '' ? $val : null;
    }
}

// ---------- output ----------
$out = [
    "url" => $url,
    "http" => [
        "status" => $info["http_code"] ?? null,
        "content_type" => $info["content_type"] ?? null
    ],
    "product" => $product,
    "debug" => [
        "jsonld_found" => count($jsonldAll),
        "og_images_found" => count($metas["og:image"]),
    ]
];

respond($out, 200);