<?php
/**
 * masterkala_daste.php (v1.2 - fixed)
 * Category Web Service for masterkala.com
 * - پشتیبانی دقیق از صفحه‌بندی BASE و BASE-Pages{N}e (شروع از هر N)
 * - استخراج نام از alt تصویر کارت محصول، و درصورت نبود از متن لینک
 * - DOMXPath + fallback Regex
 * PHP 7+
 *
 * مثال‌ها:
 *   ?url=https://masterkala.com/categories/163/products/Wearable-Gadget&pages=3
 *   ?url=https://masterkala.com/categories/163/products/Wearable-Gadget-Pages2e&pages=2
 */

ini_set('display_errors', 0);
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
mb_internal_encoding('UTF-8');

function respond($arr){ echo json_encode($arr, JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT); exit; }
function err($msg){ respond(['ok'=>false,'error'=>$msg]); }

$url   = isset($_GET['url'])   ? trim($_GET['url'])   : '';
$pages = isset($_GET['pages']) ? (int)$_GET['pages']  : 1;

if(!$url)  err('invalid_or_missing_url');
if($pages < 1) $pages = 1;
if(!preg_match('~^https?://([^/]*\.)?masterkala\.com/.*~i', $url)) err('invalid_host_only_masterkala.com');

/* ---------- HTTP ---------- */
function http_get($u, $timeout=40){
  $ctx = stream_context_create([
    'http'=>[
      'method'=>'GET',
      'timeout'=>$timeout,
      'header'=>
        "User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 ".
        "(KHTML, like Gecko) Chrome/124.0 Safari/537.36\r\n".
        "Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8\r\n",
      'ignore_errors'=>true
    ]
  ]);
  $h = @file_get_contents($u, false, $ctx);
  if($h === false || $h === null) return null;
  if(function_exists('mb_detect_encoding')){
    $enc = mb_detect_encoding($h, 'UTF-8, ISO-8859-1, Windows-1256, Windows-1252', true);
    if($enc && strtoupper($enc) !== 'UTF-8'){
      $h = @mb_convert_encoding($h, 'UTF-8', $enc);
    }
  }
  return $h;
}

/* ---------- URL Utils ---------- */
function make_abs_url($base, $maybe){
  if(preg_match('~^https?://~i', $maybe)) return $maybe;
  if(strpos($maybe, '//')===0){
    $scheme = parse_url($base, PHP_URL_SCHEME) ?: 'https';
    return $scheme . ':' . $maybe;
  }
  $p = parse_url($base);
  $scheme = $p['scheme'] ?? 'https';
  $host   = $p['host'] ?? '';
  $root   = $scheme.'://'.$host.(isset($p['port'])?':'.$p['port']:'');
  if(strpos($maybe, '/')===0) return $root.$maybe;
  $path = isset($p['path']) ? preg_replace('~/[^/]*$~','/',$p['path']) : '/';
  return $root.$path.$maybe;
}

/**
 * صفحه‌بندی دقیق:
 * - اگر ورودی BASE باشد → [BASE(=p1), BASE-Pages2e, BASE-Pages3e, ...]
 * - اگر ورودی خودش BASE-PagesNe باشد → [PagesNe, Pages(N+1)e, ...] (برای p1 به BASE برمی‌گردانیم)
 */
function build_page_urls_exact($input_url, $pages){
  $base = $input_url;
  $startN = 1;
  // آیا -Pages{N}e در ورودی هست؟
  if(preg_match('~^(.*?)(?:-Pages(\d+)e)/*$~i', rtrim($input_url,'/'), $mm)){
    $base   = $mm[1];
    $startN = max(2, (int)$mm[2]); // اگر -Pages1e داده باشند، عملاً همان p1 است
  } else {
    $base = rtrim($base, '/');
    $startN = 1;
  }

  $urls = [];
  if($startN === 1){
    // صفحه 1
    $urls[] = $base;
    // صفحات 2..N
    for($i=2; $i<= $pages; $i++){
      $urls[] = $base . '-Pages'.$i.'e';
    }
  } else {
    // از N شروع می‌کنیم و به تعداد pages می‌سازیم
    $end = $startN + $pages - 1;
    for($i=$startN; $i<= $end; $i++){
      if($i === 1) $urls[] = $base;
      else $urls[] = $base . '-Pages'.$i.'e';
    }
  }
  return $urls;
}

/* ---------- Extractors ---------- */
function extract_products_from_html($html, $base_url){
  $out = [];

  // DOMXPath
  $dom = new DOMDocument();
  libxml_use_internal_errors(true);
  $dom->loadHTML('<?xml encoding="utf-8"?>'.$html, LIBXML_NOWARNING|LIBXML_NOERROR);
  libxml_clear_errors();
  $xp = new DOMXPath($dom);

  // هر <a> که به /product/{id}/{slug} برود
  $nodes = $xp->query('//a[contains(@href,"/product/")]');
  if($nodes && $nodes->length){
    foreach($nodes as $a){
      $href = $a->getAttribute('href');
      if(!$href) continue;
      if(!preg_match('~/(?:product)/\d+/.+~i', $href)) continue;

      // نام از alt تصویر (دقیق‌ترین متن عنوان در کارت)
      $name = '';
      $imgs = $a->getElementsByTagName('img');
      if($imgs && $imgs->length){
        $alt = trim($imgs->item(0)->getAttribute('alt'));
        if($alt !== '') $name = preg_replace('/\s+/u',' ', $alt);
      }
      // اگر alt نبود، از متن لینک
      if($name === ''){
        $txt = trim(preg_replace('/\s+/u',' ', $a->textContent));
        if($txt !== '') $name = $txt;
      }
      if($name === '') $name = 'بدون نام';

      $abs = make_abs_url($base_url, $href);
      $out[] = ['name'=>$name, 'url'=>$abs];
    }
  }

  // fallback Regex (درصورت شکست DOM)
  if(empty($out)){
    if(preg_match_all('~<a[^>]+href=["\']([^"\']*/product/\d+/[^"\']+)["\'][^>]*>(.*?)</a>~is', $html, $m, PREG_SET_ORDER)){
      foreach($m as $mm){
        $abs = make_abs_url($base_url, html_entity_decode($mm[1], ENT_QUOTES|ENT_SUBSTITUTE, 'UTF-8'));
        // سعی برای گرفتن alt اگر داخل a تصویر هست
        $alt = '';
        if(preg_match('~<img[^>]+alt=["\']([^"\']+)["\']~i', $mm[0], $am)){
          $alt = trim($am[1]);
        }
        $txt = trim(preg_replace('/\s+/u',' ', strip_tags($mm[2])));
        $name = $alt ?: ($txt!==''?$txt:'بدون نام');
        $out[] = ['name'=>$name, 'url'=>$abs];
      }
    }
  }

  // یکتا بر اساس URL
  $seen = []; $final = [];
  foreach($out as $it){
    $k = strtolower($it['url']);
    if(isset($seen[$k])) continue;
    $seen[$k] = 1; $final[] = $it;
  }
  return $final;
}

/* ---------- Runner ---------- */
$page_urls = build_page_urls_exact($url, $pages);

$products = [];
$debug = [];
foreach($page_urls as $idx => $purl){
  $h = http_get($purl);
  $found = 0;
  if($h){
    $list = extract_products_from_html($h, $purl);
    $found = count($list);
    if($found) $products = array_merge($products, $list);
  }
  $debug[] = ['page'=>$idx+1, 'url'=>$purl, 'found'=>$found];
}

// یکتاسازی نهایی
$seen = []; $uniq = [];
foreach($products as $it){
  $k = strtolower($it['url']);
  if(isset($seen[$k])) continue;
  $seen[$k]=1; $uniq[]=$it;
}

respond([
  'ok'       => true,
  'url'      => rtrim($url,'/'),
  'pages'    => $pages,
  'count'    => count($uniq),
  'products' => $uniq,
  'debug'    => $debug
]);