<?php
// irtcishop.php — Single product scraper for irtcishop.net (PHP 7+)
declare(strict_types=1);
mb_internal_encoding('UTF-8');
header('Content-Type: application/json; charset=utf-8');
ini_set('display_errors','0'); error_reporting(E_ALL);

try {
  $u = isset($_GET['url']) ? trim($_GET['url']) : '';
  if (!$u) { http_response_code(400); echo j(['ok'=>false,'error'=>'missing_url']); exit; }
  if (strpos($u,'http')!==0) $u = 'https://'.$u;

  // ✅ فقط irtcishop.net مجاز است
  $host = parse_url($u, PHP_URL_HOST) ?: '';
  if (!preg_match('/(^|\.)irtcishop\.net$/i', $host)) {
    http_response_code(400); echo j(['ok'=>false,'error'=>'only_irtcishop.net_allowed']); exit;
  }

  $html = fetch($u);
  if (!$html) { echo j(['ok'=>false,'error'=>'fetch_failed']); exit; }

  // DOM + XPath
  libxml_use_internal_errors(true);
  $dom = new DOMDocument('1.0','UTF-8');
  // اطمینان از charset
  if (stripos($html,'<meta charset=')===false) $html = '<meta charset="utf-8">'.$html;
  $dom->loadHTML($html, LIBXML_NOWARNING|LIBXML_NOERROR);
  libxml_clear_errors();
  $xp = new DOMXPath($dom);

  // === نام محصول
  $name = tx($xp, '//h1[contains(@class,"product_title")]')
       ?: meta($xp,'og:title')
       ?: null;

  // === توضیحات (HTML + متن)
  $desc_html = inner($xp, '//*[@id="tab-description"]')
            ?: inner($xp, '//div[contains(@class,"product-short-description")]');
  if (!$desc_html) {
    // بکاپ از JSON-LD Product.description
    $ld = parse_ld_json($xp);
    if (isset($ld['Product']['description'])) $desc_html = trim((string)$ld['Product']['description']);
  }
  $desc_text = $desc_html ? trim(normalize_ws(strip_tags($desc_html))) : null;

  // === قیمت (اول OG، بعد JSON-LD، بعد UI)
  $price_raw = meta($xp,'product:price:amount');
  $currency  = meta($xp,'product:price:currency');
  if ($price_raw===null || $currency===null) {
    $ld = isset($ld)?$ld:parse_ld_json($xp);
    if (isset($ld['Product']['offers']['price']))       $price_raw = (string)$ld['Product']['offers']['price'];
    if (isset($ld['Product']['offers']['priceCurrency'])) $currency  = (string)$ld['Product']['offers']['priceCurrency'];
  }
  if ($price_raw===null) {
    // fallback: خواندن عدد از توی قیمت UI (در صورت موجود)
    $price_raw = norm_num(tx($xp,'(//span[contains(@class,"amount")]//bdi)[1]'));
  }
  $price_toman = $price_raw!==null ? to_toman($price_raw, $currency ?: 'IRR') : null;

  // === موجودی / وضعیت
  $stock_text = tx($xp, '//p[contains(@class,"stock")]') ?: null;

  // === تصاویر: گالری + JSON-LD
  $images = [];
  foreach ($xp->query('//div[contains(@class,"woocommerce-product-gallery")]//a[@href]') as $a) {
    $href = $a->getAttribute('href'); if ($href) $images[] = absurl($href,$u);
  }
  if (empty($images)) {
    foreach ($xp->query('//div[contains(@class,"woocommerce-product-gallery")]//img') as $img) {
      $src = $img->getAttribute('data-large_image') ?: $img->getAttribute('src') ?: $img->getAttribute('data-src');
      if ($src) $images[] = absurl($src,$u);
    }
  }
  // JSON-LD image[]
  $ld = isset($ld)?$ld:parse_ld_json($xp);
  if (isset($ld['Product']['image']) && is_array($ld['Product']['image'])) {
    foreach ($ld['Product']['image'] as $im) {
      if (is_array($im) && !empty($im['url'])) $images[] = absurl($im['url'],$u);
      elseif (is_string($im)) $images[] = absurl($im,$u);
    }
  }
  $images = array_values(array_unique($images));

  // === SKU
  $sku = tx($xp, '//*[contains(@class,"sku")]')
      ?: (isset($ld['Product']['sku']) ? (string)$ld['Product']['sku'] : null);

  // === برند (در صورت وجود افزونه YITH Brands)
  $brand = tx($xp, '//span[contains(@class,"yith-wcbr-brands")]//a');

  // === دسته‌ها
  $categories = [];
  foreach ($xp->query('//span[contains(@class,"posted_in")]//a') as $a) {
    $categories[] = trim($a->textContent);
  }
  $categories = array_values(array_unique(array_filter($categories)));

  // === ویژگی‌ها
  $attributes = [];
  foreach ($xp->query('//table[contains(@class,"woocommerce-product-attributes")]//tr') as $tr) {
    $label = tx($xp,'.//th',$tr); $value = tx($xp,'.//td',$tr);
    if ($label && $value) $attributes[] = ['name'=>$label,'value'=>normalize_ws($value)];
  }

  // === خروجی
  echo j([
    'ok'               => true,
    'url'              => $u,
    'name'             => $name,
    'description_html' => $desc_html,
    'description_text' => $desc_text,
    'price_toman'      => $price_toman,
    'currency_source'  => $currency ?: 'IRR',
    'price_raw'        => $price_raw,
    'stock'            => $stock_text,
    'images'           => $images,
    'sku'              => $sku,
    'brand'            => $brand,
    'categories'       => $categories,
    'attributes'       => $attributes,
    'debug'            => [
      'og_price_amount'   => meta($xp,'product:price:amount'),
      'og_price_currency' => meta($xp,'product:price:currency'),
    ]
  ]);
} catch (Throwable $e) {
  http_response_code(500);
  echo j(['ok'=>false,'error'=>'internal_error','msg'=>$e->getMessage()]);
}

/* ===== Helpers ===== */
function j($a){ return json_encode($a, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES|JSON_PRETTY_PRINT); }
function fetch(string $url, int $t=25): ?string {
  if (!function_exists('curl_init')) {
    $ctx = stream_context_create(['http'=>['timeout'=>$t,'header'=>"User-Agent:Mozilla/5.0\r\n"]]);
    $b = @file_get_contents($url,false,$ctx); return $b?:null;
  }
  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER=>true, CURLOPT_FOLLOWLOCATION=>true, CURLOPT_MAXREDIRS=>5,
    CURLOPT_TIMEOUT=>$t, CURLOPT_CONNECTTIMEOUT=>15,
    CURLOPT_SSL_VERIFYPEER=>true, CURLOPT_SSL_VERIFYHOST=>2,
    CURLOPT_USERAGENT=>'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 Chrome/126 Safari/537.36'
  ]);
  $b = curl_exec($ch); $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE); curl_close($ch);
  return ($b!==false && $code<400) ? $b : null;
}
function tx(DOMXPath $xp, string $q, DOMNode $ctx=null): ?string {
  $n = $ctx ? $xp->query($q,$ctx)->item(0) : $xp->query($q)->item(0);
  if ($n){ $t = trim($n->textContent); return $t!=='' ? $t : null; }
  return null;
}
function inner(DOMXPath $xp, string $q, DOMNode $ctx=null): ?string {
  $n = $ctx ? $xp->query($q,$ctx)->item(0) : $xp->query($q)->item(0);
  if (!$n) return null;
  $h=''; foreach ($n->childNodes as $c){ $h .= $n->ownerDocument->saveHTML($c); }
  $h = trim($h);
  return $h !== '' ? $h : null;
}
function meta(DOMXPath $xp, string $k): ?string {
  foreach ([
    "//meta[@property='{$k}']/@content",
    "//meta[@name='{$k}']/@content"
  ] as $q) {
    $n = $xp->query($q)->item(0);
    if ($n){ $v = trim($n->nodeValue); if ($v!=='') return $v; }
  }
  return null;
}
function norm_num(?string $s): ?string {
  if ($s===null) return null;
  $map = ['۰'=>'0','۱'=>'1','۲'=>'2','۳'=>'3','۴'=>'4','۵'=>'5','۶'=>'6','۷'=>'7','۸'=>'8','۹'=>'9','٬'=>'','٫'=>'.',','=>''];
  $s = strtr($s,$map);
  $s = preg_replace('/[^\d\.]/','',$s);
  return $s!=='' ? $s : null;
}
function to_toman($raw, string $cur='IRR'): int {
  $v = (float)$raw;
  $c = strtoupper($cur);
  if ($c==='IRR') $v /= 10; // ریال → تومان
  return (int)round($v);
}
function absurl(string $maybe, string $base): string {
  if ($maybe==='') return '';
  if (strpos($maybe,'//')===0) {
    $sch = parse_url($base,PHP_URL_SCHEME) ?: 'https'; return $sch.':'.$maybe;
  }
  if (parse_url($maybe, PHP_URL_SCHEME)) return $maybe;
  $p = parse_url($base);
  $root = $p['scheme'].'://'.$p['host'].(isset($p['port'])?':'.$p['port']:'');
  if (strpos($maybe,'/')===0) return $root.$maybe;
  $dir = $root.rtrim(dirname($p['path']),'/').'/';
  return $dir.$maybe;
}
function normalize_ws(string $s): string {
  return trim(preg_replace('/\s+/u',' ',$s));
}
function parse_ld_json(DOMXPath $xp): array {
  $data = ['Product'=>[]];
  foreach ($xp->query('//script[@type="application/ld+json"]') as $sc) {
    $json = trim($sc->textContent);
    if ($json==='') continue;
    $obj = json_decode($json, true);
    if (!is_array($obj)) continue;
    // RankMath graph style
    if (isset($obj['@graph']) && is_array($obj['@graph'])) {
      foreach ($obj['@graph'] as $node) {
        if (isset($node['@type']) && ( $node['@type']==='Product' || (is_array($node['@type']) && in_array('Product',$node['@type'],true)) )) {
          $data['Product'] = $node + ($data['Product'] ?? []);
        }
      }
    } elseif (isset($obj['@type']) && ($obj['@type']==='Product' || (is_array($obj['@type']) && in_array('Product',$obj['@type'],true)))) {
      $data['Product'] = $obj + ($data['Product'] ?? []);
    }
  }
  return $data;
}