<?php
/**
 * GPGSM Category Products Webservice (UTF-8 & IDN safe)
 * PHP 7+
 * GET:
 *   - url   : category URL (e.g. https://gpgsm.ir/product-category/ابزار-ها/هویه-ها/)
 *   - pages : number of pages to scan (>=1)
 * Output: JSON { ok, source, count, products:[{name,url,page}], errors }
 */

declare(strict_types=1);
mb_internal_encoding('UTF-8');
header('Content-Type: application/json; charset=utf-8');

// ---------- helpers ----------
function respond($data, int $code = 200) {
    http_response_code($code);
    echo json_encode($data, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES|JSON_PRETTY_PRINT);
    exit;
}

/**
 * نرمال‌سازی URL پایهٔ دسته:
 * - حذف /page/N/ از انتهای *مسیر*
 * - درصد-کدنویسی segment-wise برای اسلاگ‌های فارسی/یونیکد
 * - پشتیبانی از دامنه‌های یونیکد (IDN → Punycode) در صورت نصب intl
 */
function normalize_base_category_url(string $url): string {
    $url = trim($url);
    if ($url === '') return '';

    $p = parse_url($url);
    if (!$p || empty($p['scheme']) || empty($p['host'])) return '';

    $scheme = strtolower($p['scheme']);
    $host   = $p['host'];
    $port   = isset($p['port']) ? (int)$p['port'] : 0;
    $path   = isset($p['path']) ? $p['path'] : '/';

    // حذف /page/N/ از انتهای *مسیر*
    $path = preg_replace('~(/page/\d+/?)+$~i', '/', $path);

    // تضمین اسلش ابتدایی و انتهایی
    if ($path === '' || $path[0] !== '/') $path = '/' . ltrim($path, '/');
    if (substr($path, -1) !== '/') $path .= '/';

    // اگر مسیر فارسی/خام است، هر سگمنت را جداگانه rawurlencode کن
    $segments = array_filter(explode('/', trim($path, '/')), static function($s){ return $s !== ''; });
    $encSegs  = [];
    foreach ($segments as $seg) {
        // اگر قبلاً percent-encoded است، همان را نگه دار؛ وگرنه encode
        if (preg_match('/%[0-9A-Fa-f]{2}/', $seg)) {
            $encSegs[] = $seg;
        } else {
            $encSegs[] = rawurlencode($seg);
        }
    }
    $path = '/' . implode('/', $encSegs) . '/';

    // دامنهٔ یونیکد → ASCII (پانی‌کد) اگر intl موجود باشد
    if (function_exists('idn_to_ascii')) {
        $host_ascii = idn_to_ascii($host, IDNA_DEFAULT, INTL_IDNA_VARIANT_UTS46);
        if ($host_ascii) $host = $host_ascii;
    }

    $portStr = $port ? ':' . $port : '';
    return $scheme . '://' . $host . $portStr . $path;
}

/** ساخت URL صفحه nام؛ صفحهٔ ۱ همان base */
function build_page_url(string $base, int $n): string {
    return ($n <= 1) ? $base : (rtrim($base, '/') . '/page/' . $n . '/');
}

/** دریافت HTML با cURL (با پشتیبانی از gzip/deflate) */
function fetch_html(string $url, int $timeout = 20) {
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL            => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_MAXREDIRS      => 5,
        CURLOPT_CONNECTTIMEOUT => $timeout,
        CURLOPT_TIMEOUT        => $timeout,
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_SSL_VERIFYHOST => 2,
        CURLOPT_ENCODING       => '', // gzip/deflate/brotli (اگر سرور پشتیبانی کند)
        CURLOPT_USERAGENT      => 'MajaziGPGSMCategoryWS/1.2 (+php; crawler)',
        CURLOPT_HTTPHEADER     => ['Accept: text/html,application/xhtml+xml;q=0.9,*/*;q=0.8'],
    ]);
    $body = curl_exec($ch);
    $err  = curl_error($ch);
    $code = (int)curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
    curl_close($ch);
    if ($body === false || $code >= 400) {
        return ['ok' => false, 'error' => $err ?: ('HTTP '.$code), 'code' => $code];
    }
    return ['ok' => true, 'html' => $body, 'code' => $code];
}

/** تبدیل href نسبی به مطلق براساس base */
function make_absolute_url(string $href, string $base): string {
    if (preg_match('~^https?://~i', $href)) return $href;
    $bp = parse_url($base);
    if (!$bp || empty($bp['scheme']) || empty($bp['host'])) return $href;
    $scheme = $bp['scheme'];
    $host   = $bp['host'];
    $port   = isset($bp['port']) ? ':' . $bp['port'] : '';
    if (strpos($href, '//') === 0) return $scheme . ':' . $href;
    if (substr($href, 0, 1) === '/') return $scheme . '://' . $host . $port . $href;

    $path = isset($bp['path']) ? $bp['path'] : '/';
    $path = preg_replace('~/[^/]*$~', '/', $path);
    $abs  = $path . $href;

    // resolve ../ و ./
    $parts = [];
    foreach (explode('/', $abs) as $seg) {
        if ($seg === '' || $seg === '.') continue;
        if ($seg === '..') { array_pop($parts); continue; }
        $parts[] = $seg;
    }
    return $scheme . '://' . $host . $port . '/' . implode('/', $parts) . (substr($href, -1) === '/' ? '/' : '');
}

/** تنها URLهای محصول روی همان دامنه و مسیر /product/ مجازند (حذف share links و خارجی‌ها) */
function is_product_url(string $href, string $baseHost): bool {
    $p = parse_url($href);
    if (!$p || empty($p['scheme']) || empty($p['host']) || empty($p['path'])) return false;
    $h = strtolower($p['host']);
    $b = strtolower($baseHost);
    $same = ($h === $b) || (substr($h, -strlen('.'.$b)) === '.'.$b);
    if (!$same) return false;
    return strpos($p['path'], '/product/') === 0;
}

/** استخراج محصولات از HTML: انتخابگر استاندارد ووکامرس در GPGSM */
function extract_products_from_html(string $html, string $baseUrl): array {
    libxml_use_internal_errors(true);
    $doc  = new DOMDocument();
    $html = mb_convert_encoding($html, 'HTML-ENTITIES', 'UTF-8');
    $doc->loadHTML($html);
    $xp   = new DOMXPath($doc);

    $baseHost = parse_url($baseUrl, PHP_URL_HOST) ?: '';
    $seen = [];
    $out  = [];

    // لینک‌های کارت محصول ووکامرس
    $queries = [
        "//a[contains(@class,'woocommerce-LoopProduct-link') and contains(@class,'woocommerce-loop-product__link') and @href]",
        "//li[contains(@class,'product')]//a[contains(@class,'woocommerce-LoopProduct-link') and @href]",
        "//div[contains(@class,'product')]//a[contains(@class,'woocommerce-LoopProduct-link') and @href]"
    ];

    foreach ($queries as $q) {
        foreach ($xp->query($q) as $a) {
            /** @var DOMElement $a */
            $href = trim($a->getAttribute('href'));
            if ($href === '') continue;
            $href = html_entity_decode($href, ENT_QUOTES, 'UTF-8');
            $hrefAbs = make_absolute_url($href, $baseUrl);
            if (!is_product_url($hrefAbs, $baseHost)) continue;

            $name = trim($a->textContent);
            if ($name === '') {
                // fallback: عنوان نزدیک در h2/h3
                $h = $a->parentNode;
                while ($h && $h->nodeType === XML_ELEMENT_NODE) {
                    $try = null;
                    $hs2 = $h->getElementsByTagName('h2');
                    if ($hs2->length > 0) $try = trim($hs2->item(0)->textContent);
                    if (!$try) {
                        $hs3 = $h->getElementsByTagName('h3');
                        if ($hs3->length > 0) $try = trim($hs3->item(0)->textContent);
                    }
                    if ($try) { $name = $try; break; }
                    $h = $h->parentNode;
                }
            }
            $name = preg_replace('/\s{2,}/u', ' ', $name);
            if ($name === '') continue;

            $key = strtolower($hrefAbs);
            if (!isset($seen[$key])) {
                $out[] = ['name' => $name, 'url' => $hrefAbs];
                $seen[$key] = true;
            }
        }
    }

    return $out;
}

// ---------- inputs ----------
$baseUrl = isset($_GET['url']) ? (string)$_GET['url'] : '';
$pages   = isset($_GET['pages']) ? (int)$_GET['pages'] : 0;

if ($baseUrl === '' || $pages <= 0) {
    respond([
        'ok' => false,
        'error' => 'پارامترهای ورودی نامعتبرند. استفاده: ?url=https://gpgsm.ir/product-category/ابزار-ها/هویه-ها/&pages=2'
    ], 400);
}

$baseUrl = normalize_base_category_url($baseUrl);
if ($baseUrl === '') {
    respond(['ok'=>false,'error'=>'URL نامعتبر است. باید با http/https شروع شود.'], 400);
}

// ---------- crawl ----------
$products = [];
$errors   = [];

for ($i = 1; $i <= $pages; $i++) {
    $pageUrl = build_page_url($baseUrl, $i);
    $res = fetch_html($pageUrl, 25);
    if (!$res['ok']) {
        $errors[] = ['page'=>$i, 'url'=>$pageUrl, 'error'=>$res['error'] ?? ('HTTP '.$res['code'])];
        continue;
    }
    $items = extract_products_from_html($res['html'], $baseUrl);
    foreach ($items as $it) {
        $products[] = ['name'=>$it['name'], 'url'=>$it['url'], 'page'=>$i];
    }
}

// dedup by URL
$seen = []; $out = [];
foreach ($products as $p) {
    $k = strtolower($p['url']);
    if (!isset($seen[$k])) { $seen[$k]=1; $out[]=$p; }
}

// ---------- response ----------
respond([
    'ok' => true,
    'source' => [
        'base_url' => $baseUrl,
        'pages'    => $pages,
        'note'     => 'Paging format assumed as /page/{n}/; page 1 uses base URL.'
    ],
    'count'    => count($out),
    'products' => $out,
    'errors'   => $errors,
]);
