<?php
/**
 * GPGSM Product Info JSON API — v1.2.0
 * PHP 7.4+
 *
 * Usage:
 *   gpgsm_product_api.php?url=<product_url>&pretty=1
 *   Optional: &currency=toman|rial  (default=toman)
 *            &strip_brand=1         (remove site branding words)
 */

declare(strict_types=1);

// ---------------------- Utils ----------------------
function respond_json($data, int $code = 200, bool $pretty = false): void {
    http_response_code($code);
    header('Content-Type: application/json; charset=utf-8');
    header('Access-Control-Allow-Origin: *');
    $opt = JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES;
    if ($pretty) $opt |= JSON_PRETTY_PRINT;
    echo json_encode($data, $opt);
    exit;
}
function error_json(string $message, int $code = 400, array $meta = []): void {
    respond_json(['ok' => false, 'error' => $message, 'meta' => $meta], $code, true);
}
function fa2en(?string $s): ?string {
    if ($s === null) return null;
    $fa = ['۰','۱','۲','۳','۴','۵','۶','۷','۸','۹','٠','١','٢','٣','٤','٥','٦','٧','٨','٩'];
    $en = ['0','1','2','3','4','5','6','7','8','9','0','1','2','3','4','5','6','7','8','9'];
    return str_replace($fa, $en, $s);
}
function normalize_price_to_int(?string $s): ?int {
    if ($s === null) return null;
    $s = fa2en($s);
    $s = preg_replace('/[^\d]/u', '', $s);
    if ($s === '' || $s === null) return null;
    return intval($s, 10);
}
function decode_jsonld(string $json): array {
    $json = trim($json);
    $json = preg_replace('/<!--.*?-->/s', '', $json) ?? $json;
    $json = preg_replace('/,\s*([}\]])/', '$1', $json) ?? $json;
    $x = json_decode($json, true);
    if (json_last_error() === JSON_ERROR_NONE && is_array($x)) return $x;
    $x = json_decode(utf8_encode($json), true);
    return (json_last_error() === JSON_ERROR_NONE && is_array($x)) ? $x : [];
}
function curl_fetch(string $url, array $headers = [], bool $withHeaders = true, int $timeout = 25): array {
    $ch = curl_init();
    $ua = 'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120 Safari/537.36 ProductAPIBot/1.2';
    $base = [
        'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
        'Accept-Language: fa-IR,fa;q=0.9,en-US;q=0.8,en;q=0.7',
        'Cache-Control: no-cache',
        'Pragma: no-cache',
        'User-Agent: ' . $ua,
    ];
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_HTTPHEADER => array_merge($base, $headers),
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_MAXREDIRS => 5,
        CURLOPT_CONNECTTIMEOUT => 15,
        CURLOPT_TIMEOUT => 25,
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_SSL_VERIFYHOST => 2,
        CURLOPT_ENCODING => '',
        CURLOPT_HEADER => $withHeaders,
    ]);
    $resp = curl_exec($ch);
    if ($resp === false) {
        $err = curl_error($ch);
        curl_close($ch);
        return [null, ['error' => 'curl_error', 'detail' => $err]];
    }
    $status = curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
    $final  = curl_getinfo($ch, CURLINFO_EFFECTIVE_URL);
    $hsize  = $withHeaders ? curl_getinfo($ch, CURLINFO_HEADER_SIZE) : 0;
    curl_close($ch);
    if ($withHeaders) {
        $rawHeaders = substr($resp, 0, $hsize);
        $body = substr($resp, $hsize);
        return [$body, ['status' => $status, 'final_url' => $final, 'headers' => $rawHeaders]];
    }
    return [$resp, ['status' => $status, 'final_url' => $final]];
}
function fetch_content(string $url): array {
    if (!preg_match('#^https?://#i', $url)) {
        if (!is_readable($url)) return [null, ['source'=>'local', 'error'=>'file_not_readable']];
        $html = file_get_contents($url);
        return [$html === false ? null : $html, ['source'=>'local']];
    }
    list($html, $meta) = curl_fetch($url, [], true, 25);
    $meta['source'] = 'remote';
    return [$html, $meta];
}
function dom_and_xpath(string $html): array {
    libxml_use_internal_errors(true);
    $dom = new DOMDocument();
    $clean = preg_replace('#<script\b[^>]*>.*?</script>#is', '', $html) ?? $html;
    $dom->loadHTML($clean);
    $xp = new DOMXPath($dom);
    return [$dom, $xp];
}
function meta_get(DOMXPath $xp, string $attr, string $val): ?string {
    $attr = strtolower($attr);
    $q = sprintf('//meta[translate(@%1$s,"ABCDEFGHIJKLMNOPQRSTUVWXYZ","abcdefghijklmnopqrstuvwxyz")="%2$s"]/@content', $attr, $val);
    $n = $xp->query($q);
    return ($n && $n->length > 0) ? trim($n->item(0)->nodeValue) : null;
}
function node_text(DOMXPath $xp, string $xpath): ?string {
    $n = $xp->query($xpath);
    if (!$n || $n->length === 0) return null;
    $t = trim($n->item(0)->textContent);
    return $t === '' ? null : $t;
}
function nodes_attr(DOMXPath $xp, string $xpath, string $attr = 'src'): array {
    $out = [];
    $list = $xp->query($xpath);
    if ($list) foreach ($list as $node) {
        if ($node->hasAttribute($attr)) {
            $v = trim($node->getAttribute($attr));
            if ($v !== '') $out[] = $v;
        } elseif ($node->nodeType === XML_ATTRIBUTE_NODE) {
            $out[] = trim($node->nodeValue);
        }
    }
    return array_values(array_unique($out));
}
function cleanup_whitespace(?string $html): ?string {
    if ($html === null) return null;
    return trim(preg_replace('/\s+/', ' ', $html));
}

// ---------------------- Extractors ----------------------
function extract_jsonld_blocks(DOMXPath $xp): array {
    $out = [];
    $nodes = $xp->query('//script[@type="application/ld+json"]');
    if (!$nodes) return $out;
    foreach ($nodes as $s) {
        $data = decode_jsonld($s->textContent ?? '');
        if (!$data) continue;
        $out[] = $data;
    }
    return $out;
}
function first_product_from_jsonld(array $blocks): ?array {
    foreach ($blocks as $b) {
        $cands = [];
        if (isset($b['@type'])) $cands = [$b];
        elseif (isset($b['@graph']) && is_array($b['@graph'])) $cands = $b['@graph'];
        elseif (is_array($b)) $cands = $b;
        foreach ($cands as $it) {
            if (!is_array($it)) continue;
            $t = $it['@type'] ?? null;
            $isProduct = is_array($t) ? in_array('Product', $t, true) : (strcasecmp((string)$t, 'Product') === 0);
            if ($isProduct) return $it;
        }
    }
    return null;
}
function extract_product_from_jsonld(array $p): array {
    $g = function($a,$k,$d=null){ return $a[$k] ?? $d; };
    $offer = $g($p,'offers',[]);
    if (is_array($offer) && isset($offer[0])) $offer = $offer[0];

    $imgs = [];
    if (isset($p['image'])) {
        if (is_array($p['image'])) foreach ($p['image'] as $im) {
            if (is_array($im) && isset($im['url'])) $imgs[] = $im['url'];
            elseif (is_string($im)) $imgs[] = $im;
        } elseif (is_string($p['image'])) $imgs[] = $p['image'];
    }
    $brand = $g($p,'brand');
    if (is_array($brand)) $brand = $brand['name'] ?? ($brand['@id'] ?? null);

    return [
        'name'        => $g($p,'name'),
        'description' => $g($p,'description'),
        'sku'         => $g($p,'sku'),
        'brand'       => $brand,
        'category'    => $g($p,'category'),
        'images'      => array_values(array_unique($imgs)),
        'offers'      => [
            'price'         => normalize_price_to_int((string)($offer['price'] ?? '')),
            'priceCurrency' => $offer['priceCurrency'] ?? null,
            'availability'  => $offer['availability'] ?? null,
            'itemCondition' => $offer['itemCondition'] ?? null,
            'url'           => $offer['url'] ?? null,
        ],
        'aggregateRating' => (isset($p['aggregateRating']) && is_array($p['aggregateRating'])) ? [
            'ratingValue' => (float)($p['aggregateRating']['ratingValue'] ?? 0),
            'reviewCount' => (int)($p['aggregateRating']['reviewCount'] ?? 0),
        ] : null,
    ];
}
function extract_from_meta(DOMXPath $xp): array {
    $og = [
        'title'        => meta_get($xp,'property','og:title'),
        'description'  => meta_get($xp,'property','og:description'),
        'image'        => meta_get($xp,'property','og:image'),
        'url'          => meta_get($xp,'property','og:url'),
        'site_name'    => meta_get($xp,'property','og:site_name'),
        'type'         => meta_get($xp,'property','og:type'),
        'updated_time' => meta_get($xp,'property','og:updated_time'),
        'locale'       => meta_get($xp,'property','og:locale'),
    ];
    $tw = [
        'card'  => meta_get($xp,'name','twitter:card'),
        'title' => meta_get($xp,'name','twitter:title'),
        'description' => meta_get($xp,'name','twitter:description'),
        'image' => meta_get($xp,'name','twitter:image'),
        'label1'=> meta_get($xp,'name','twitter:label1'),
        'data1' => meta_get($xp,'name','twitter:data1'),
        'label2'=> meta_get($xp,'name','twitter:label2'),
        'data2' => meta_get($xp,'name','twitter:data2'),
    ];
    $prod = [
        'brand'           => meta_get($xp,'property','product:brand'),
        'price:amount'    => meta_get($xp,'property','product:price:amount'),
        'price:currency'  => meta_get($xp,'property','product:price:currency'),
        'availability'    => meta_get($xp,'property','product:availability'),
        'retailer_item_id'=> meta_get($xp,'property','product:retailer_item_id'),
    ];
    $base = [
        'title'            => node_text($xp,'//title'),
        'meta_description' => meta_get($xp,'name','description'),
        'canonical'        => nodes_attr($xp,'//link[@rel="canonical"]','href')[0] ?? null,
        'lang'             => nodes_attr($xp,'//html','lang')[0] ?? null,
        'dir'              => nodes_attr($xp,'//html','dir')[0] ?? null,
    ];

    // candidates: product:price or twitter:data1 (only if really price)
    $p1 = normalize_price_to_int($prod['price:amount'] ?? null);
    $p2 = null;
    $tw_label = $tw['label1'] ? mb_strtolower(trim($tw['label1'])) : '';
    $tw_data1 = $tw['data1'] ? mb_strtolower(trim($tw['data1'])) : '';
    if (
        ($tw_label !== '' && (mb_strpos($tw_label, 'قیمت') !== false || mb_strpos($tw_label, 'price') !== false)) ||
        ($tw_data1 !== '' && (mb_strpos($tw_data1, 'تومان') !== false || mb_strpos($tw_data1, 'rial') !== false || mb_strpos($tw_data1, 'ریال') !== false))
    ) {
        $p2 = normalize_price_to_int($tw['data1']);
    }

    return [
        'base'    => $base,
        'og'      => $og,
        'twitter' => $tw,
        'product' => $prod,
        'price_candidates' => array_values(array_filter([$p1, $p2], function($v){ return $v !== null; })),
    ];
}
function parse_srcset_best(?string $srcset): ?string {
    if (!$srcset) return null;
    $best = null; $bw = -1;
    foreach (explode(',', $srcset) as $part) {
        $part = trim($part); if ($part === '') continue;
        if (preg_match('/\s+(\d+)w$/', $part, $m)) {
            $url = trim(str_replace($m[0], '', $part));
            $w = intval($m[1], 10);
            if ($w > $bw) { $bw = $w; $best = $url; }
        } else $best = preg_split('/\s+/', $part)[0];
    }
    return $best;
}
function get_wc_gallery_images(DOMXPath $xp): array {
    $urls = [];
    $nodes = $xp->query('//*[contains(@class,"woocommerce-product-gallery")]//img');
    if ($nodes) foreach ($nodes as $img) {
        $cands = [];
        foreach (['data-large_image','data-src','data-zoom-image','src'] as $a) {
            if ($img->hasAttribute($a)) {
                $v = trim($img->getAttribute($a));
                if ($v !== '') $cands[] = $v;
            }
        }
        if ($img->hasAttribute('srcset')) {
            $b = parse_srcset_best($img->getAttribute('srcset'));
            if ($b) $cands[] = $b;
        }
        foreach ($cands as $u) {
            if (preg_match('/-\d{2,3}x\d{2,3}\.(jpg|jpeg|png|webp)$/i', $u)) continue; // skip thumbs
            $urls[] = $u;
        }
    }
    $uniq = [];
    foreach ($urls as $u) if (!in_array($u, $uniq, true)) $uniq[] = $u;
    return $uniq;
}

// --------- Price Heuristics from DOM (toman) ----------
function find_price_in_summary_toman(DOMXPath $xp): ?int {
    // 1) typical WC summary area
    $n = $xp->query('(//*[contains(@class,"summary")])[1]//*[contains(@class,"price") or contains(@class,"amount") or self::bdi]');
    if ($n && $n->length > 0) {
        $txt = '';
        foreach ($n as $el) $txt .= ' ' . $el->textContent;
        $v = extract_toman_from_text($txt);
        if ($v !== null) return $v;
    }
    // 2) inside main product wrapper
    $n2 = $xp->query('(.//*[contains(@class,"product") and starts-with(@id,"product-")])[1]//*[contains(text(),"تومان") or contains(@class,"price") or contains(@class,"amount")]');
    if ($n2 && $n2->length > 0) {
        $txt = '';
        foreach ($n2 as $el) $txt .= ' ' . $el->textContent;
        $v = extract_toman_from_text($txt);
        if ($v !== null) return $v;
    }
    // 3) any element with "تومان" (fallback)
    $n3 = $xp->query('//*[contains(text(),"تومان")]');
    if ($n3 && $n3->length > 0) {
        $txt = '';
        for ($i=0; $i < min($n3->length, 8); $i++) $txt .= ' ' . $n3->item($i)->textContent; // first few matches
        $v = extract_toman_from_text($txt);
        if ($v !== null) return $v;
    }
    return null;
}
function extract_toman_from_text(string $text): ?int {
    $text = fa2en($text);
    // Find numbers near "تومان"
    if (preg_match_all('/(\d[\d,\.]*)\s*تومان/u', $text, $m)) {
        foreach ($m[1] as $num) {
            $v = intval(preg_replace('/[^\d]/', '', $num), 10);
            if ($v >= 1000) return $v; // simple sanity
        }
    }
    // Or any big number that looks like a price (toman) in the main area
    if (preg_match_all('/\d[\d,\.]{3,}/', $text, $m2)) {
        $cands = array_map(function($s){ return intval(preg_replace('/[^\d]/','',$s),10); }, $m2[0]);
        rsort($cands); // pick the largest (usually main price)
        foreach ($cands as $v) if ($v >= 1000) return $v;
    }
    return null;
}

// ---------------------- Markup extractor ----------------------
function extract_from_markup(DOMXPath $xp): array {
    $title = node_text($xp,'//*[@class="product_title" or contains(@class,"product_title")]') ?? node_text($xp,'//h1');

    // price candidates
    $priceText = node_text($xp,'//*[@class="price"]//*[contains(@class,"amount")]') ?? node_text($xp,'//*[@class="price"]');
    $priceInt  = normalize_price_to_int($priceText);

    // extra heuristics — toman directly
    $priceTomanHeur = find_price_in_summary_toman($xp);

    $sku = node_text($xp,'//*[@class="sku"]');
    $stockText = node_text($xp,'//*[contains(@class,"stock")]');

    $short = null; $n = $xp->query('//*[contains(@class,"woocommerce-product-details__short-description")]');
    if ($n && $n->length > 0) { $inner=''; foreach ($n->item(0)->childNodes as $c) $inner.=$n->item(0)->ownerDocument->saveHTML($c); $short=cleanup_whitespace($inner); }

    $desc = null; $dn = $xp->query('//*[@id="tab-description" or contains(@class,"woocommerce-Tabs-panel--description")]');
    if ($dn && $dn->length > 0) { $inner=''; foreach ($dn->item(0)->childNodes as $c) $inner.=$dn->item(0)->ownerDocument->saveHTML($c); $desc=cleanup_whitespace($inner); }

    $attrs = [];
    $rows = $xp->query('//table[contains(@class,"shop_attributes")]//tr');
    if ($rows) foreach ($rows as $tr) {
        $th = $tr->getElementsByTagName('th')->item(0);
        $td = $tr->getElementsByTagName('td')->item(0);
        if ($th && $td) {
            $name = trim($th->textContent); $value = trim($td->textContent);
            if ($name !== '' && $value !== '') $attrs[] = ['name'=>$name,'value'=>$value];
        }
    }

    $images = get_wc_gallery_images($xp);
    if (!$images) $images = nodes_attr($xp,'//img[contains(@class,"wp-post-image") or contains(@class,"attachment-woocommerce_single")]','src');

    $breadcrumbs = [];
    $crumb = $xp->query('//*[contains(@class,"woocommerce-breadcrumb")]//a');
    if ($crumb) foreach ($crumb as $a) $breadcrumbs[] = ['title'=>trim($a->textContent),'url'=>$a->getAttribute('href')];

    $categories = [];
    $cat = $xp->query('//*[contains(@class,"posted_in")]//a');
    if ($cat) foreach ($cat as $a) $categories[] = ['title'=>trim($a->textContent),'url'=>$a->getAttribute('href')];

    return [
        'title'  => $title,
        'price_text' => $priceText,
        'price_int'  => $priceInt,       // might be IRR or TOMAN depending on theme; we prefer heuristics below
        'price_toman_heur' => $priceTomanHeur,
        'sku'    => $sku,
        'stock_text' => $stockText,
        'short_description_html' => $short,
        'description_html'       => $desc,
        'attributes' => $attrs,
        'images' => $images,
        'breadcrumbs' => $breadcrumbs,
        'categories'  => $categories,
    ];
}

// ---------------------- Normalizers ----------------------
function normalize_availability(?string $v): ?string {
    if ($v === null) return null;
    $v = strtolower(trim($v));
    if (strpos($v, 'schema.org') !== false) return $v;
    $m = [
        'instock'     => 'https://schema.org/InStock',
        'in stock'    => 'https://schema.org/InStock',
        'outofstock'  => 'https://schema.org/OutOfStock',
        'out of stock'=> 'https://schema.org/OutOfStock',
        'preorder'    => 'https://schema.org/PreOrder',
        'backorder'   => 'https://schema.org/BackOrder',
    ];
    return $m[$v] ?? $v;
}
function clean_product_name(?string $name, bool $strip_brand=false): ?string {
    if ($name === null) return null;
    $name = preg_replace('/^\s*خرید\s*و\s*قیمت\s*/u', '', $name);
    if ($strip_brand) {
        $patterns = ['جرمن\s*پارت','gpgsm','جی\s*پی\s*جی\s*اس\s*ام'];
        $name = preg_replace('/\s*[-–—|]\s*(' . implode('|', $patterns) . ')\s*$/iu', '', $name);
        $name = preg_replace('/(' . implode('|', $patterns) . ')/iu', '', $name);
    }
    $name = preg_replace('/\s*[-–—]\s*$/u', '', $name);
    $name = preg_replace('/\s{2,}/u', ' ', $name);
    return trim($name);
}
function strip_brand_everywhere(string $html_or_text): string {
    $s = (string)$html_or_text;
    $patterns = ['/جرمن\s*پارت/iu','/gpgsm/iu','/جی\s*پی\s*جی\s*اس\s*ام/iu'];
    foreach ($patterns as $p) $s = preg_replace($p, '', $s);
    $s = preg_replace('/\s{2,}/u', ' ', $s);
    return trim($s);
}
function price_to_toman(?int $irr): ?int { return $irr === null ? null : intdiv($irr, 10); }
function price_display_toman(?int $toman): ?string { return $toman === null ? null : number_format($toman, 0, '.', ',') . ' تومان'; }

// ---------------------- Main ----------------------
$pretty     = isset($_GET['pretty']) && $_GET['pretty'] == '1';
$url        = $_GET['url'] ?? $_POST['url'] ?? null;
$currency   = strtolower((string)($_GET['currency'] ?? $_POST['currency'] ?? 'toman')); // 'toman' | 'rial'
$stripBrand = isset($_GET['strip_brand']) ? ($_GET['strip_brand'] == '1') : false;

if (!$url) error_json('Missing "url" parameter. Example: ?url=https://gpgsm.ir/product/...&pretty=1');

list($html, $fetchMeta) = fetch_content($url);
if ($html === null) error_json('Failed to fetch content', 502, $fetchMeta);

list($dom, $xp) = dom_and_xpath($html);

$meta    = extract_from_meta($xp);
$blocks  = extract_jsonld_blocks($xp);
$jp      = first_product_from_jsonld($blocks);
$primary = $jp ? extract_product_from_jsonld($jp) : null;
$markup  = extract_from_markup($xp);

// ----- Canonical fallback to final_url if missing -----
$canonical = $meta['base']['canonical'];
if (!$canonical && !empty($fetchMeta['final_url'])) $canonical = $fetchMeta['final_url'];

// ----- Images: JSON-LD -> OG -> Markup -----
$images = $primary['images'] ?? [];
if (!$images && !empty($meta['og']['image'])) $images[] = $meta['og']['image'];
$images = array_values(array_unique(array_merge($images, $markup['images'] ?? [])));

// ----- Price Resolution -----
$price_source = 'none';
$price_irr = $primary['offers']['price'] ?? ($meta['price_candidates'][0] ?? null); // strictly from JSON-LD/meta/twitter(filtered)
$price_dom_toman = $markup['price_toman_heur'] ?? null;

$price_currency_in = strtoupper((string)($primary['offers']['priceCurrency'] ?? ($meta['product']['price:currency'] ?? '')));

// ----- Availability -----
$availability_raw = $primary['offers']['availability'] ?? ($meta['product']['availability'] ?? null);
$availability     = normalize_availability($availability_raw);

// ----- Name & Descriptions -----
$name = $markup['title'] ?? ($primary['name'] ?? ($meta['og']['title'] ?? null));
$name = clean_product_name($name, $stripBrand);

$short_html = $markup['short_description_html'] ?? null;
$desc_html  = $markup['description_html'] ?? ($primary['description'] ?? null);
if ($stripBrand) {
    if ($short_html) $short_html = strip_brand_everywhere($short_html);
    if ($desc_html)  $desc_html  = strip_brand_everywhere($desc_html);
}

// ----- Category -----
$category = $primary['category'] ?? null;
if ($category === null && !empty($markup['breadcrumbs'])) {
    $last = end($markup['breadcrumbs']);
    $category = $last['title'] ?? null;
    reset($markup['breadcrumbs']);
}

// ----- Currency Output -----
$currency_out = ($currency === 'rial') ? 'IRR' : 'TOMAN';
$price_out = null;
$original_price = null;

// Prefer DOM (toman) when available
if ($currency_out === 'TOMAN') {
    if ($price_dom_toman !== null) {
        $price_out = $price_dom_toman;       // direct toman from page
        $price_source = 'dom';
        $original_price = ['value' => $price_out * 10, 'currency' => 'IRR'];
    } elseif ($price_irr !== null) {
        $price_out = price_to_toman((int)$price_irr);
        $price_source = 'jsonld/meta';
        $original_price = ['value' => (int)$price_irr, 'currency' => ($price_currency_in ?: 'IRR')];
    }
} else { // output IRR
    if ($price_irr !== null) {
        $price_out = (int)$price_irr;
        $price_source = 'jsonld/meta';
    } elseif ($price_dom_toman !== null) {
        $price_out = $price_dom_toman * 10;
        $price_source = 'dom';
    }
}

$price_display = null;
if ($price_out !== null) {
    $price_display = ($currency_out === 'TOMAN')
        ? price_display_toman($price_out)
        : number_format((int)$price_out, 0, '.', ',') . ' ریال';
}

// ----- Build response -----
$result = [
    'ok' => true,
    'source' => [
        'requested_url' => $url,
        'fetch' => $fetchMeta,
    ],
    'base' => [
        'title' => $meta['base']['title'],
        'meta_description' => $meta['base']['meta_description'],
        'canonical' => $canonical,
        'canonical_decoded' => $canonical ? urldecode($canonical) : null,
        'lang' => $meta['base']['lang'],
        'dir'  => $meta['base']['dir'],
    ],
    'seo_meta' => [
        'og'          => $meta['og'],
        'twitter'     => $meta['twitter'],
        'product_meta'=> $meta['product'],
    ],
    'product' => [
        'name'        => $name,
        'brand'       => $primary['brand'] ?? ($meta['product']['brand'] ?? null),
        'sku'         => $primary['sku'] ?? ($markup['sku'] ?? null),
        'category'    => $category,
        'description' => $primary['description'] ?? null,
        'short_description_html' => $short_html,
        'description_html'       => $desc_html,
        'attributes'  => $markup['attributes'] ?? null,
        'images'      => $images,
        'breadcrumbs' => $markup['breadcrumbs'] ?? null,
        'categories'  => $markup['categories'] ?? null,
    ],
    'offers' => [
        'price'            => $price_out,
        'priceCurrency'    => $currency_out,
        'price_display'    => $price_display,
        'original_price'   => $original_price, // IRR snapshot if قابل محاسبه
        'price_source'     => $price_source,   // dom | jsonld/meta | twitter | none
        'availability'     => $availability,
        'availability_raw' => $availability_raw,
        'availability_text'=> $markup['stock_text'] ?? ($meta['twitter']['data2'] ?? null),
        'itemCondition'    => $primary['offers']['itemCondition'] ?? null,
        'priceValidUntil'  => null,
        'url'              => $primary['offers']['url'] ?? ($meta['og']['url'] ?? $canonical ?? null),
    ],
    'rating' => $primary['aggregateRating'] ?? null,
    'raw' => [
        'jsonld_first_product' => $jp,
    ],
];

respond_json($result, 200, $pretty);