<?php
/**
 * Saalambabaa Product Webservice
 * PHP 7+ compatible
 *
 * Usage:
 *   sbaba.php?url=https://saalambabaa.com/SBP-8226/samsung
 */

header('Content-Type: application/json; charset=utf-8');

/* ===================== Helpers ===================== */

function sb_safe_get($array, $key, $default = null) {
    return (is_array($array) && array_key_exists($key, $array)) ? $array[$key] : $default;
}

function sb_fetch_url($url) {
    $url = trim($url);

    if (!preg_match('~^https?://~i', $url)) {
        $url = 'https://' . ltrim($url, '/');
    }

    $parts = parse_url($url);
    if (!$parts || empty($parts['host'])) {
        return [null, 'Invalid URL'];
    }

    // Optional domain lock
    if (strpos($parts['host'], 'saalambabaa.com') === false &&
        strpos($parts['host'], 'salambabaa.com') === false) {
        return [null, 'This API is limited to saalambabaa.com'];
    }

    if (function_exists('curl_init')) {
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL            => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_MAXREDIRS      => 5,
            CURLOPT_TIMEOUT        => 20,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_USERAGENT      => 'Mozilla/5.0 (compatible; sbaba-bot/1.0)',
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
        ]);
        $html = curl_exec($ch);
        $err  = curl_error($ch);
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($html === false || $code >= 400) {
            return [null, 'HTTP error: ' . $code . ' / ' . $err];
        }
        return [$html, null];
    }

    // Fallback without cURL
    $context = stream_context_create([
        'http' => [
            'method'  => 'GET',
            'timeout' => 20,
            'header'  => "User-Agent: Mozilla/5.0 (compatible; sbaba-bot/1.0)\r\n",
        ]
    ]);
    $html = @file_get_contents($url, false, $context);
    if ($html === false) {
        return [null, 'Failed to fetch HTML'];
    }
    return [$html, null];
}

/**
 * Find Product node in arbitrary JSON-LD structure.
 */
function sb_find_product_in_jsonld($data) {
    if (!is_array($data)) {
        return null;
    }
    // DFS stack search
    $stack = [$data];
    while (!empty($stack)) {
        $node = array_pop($stack);
        if (!is_array($node)) continue;

        if (isset($node['@type'])) {
            $types = is_array($node['@type']) ? $node['@type'] : [$node['@type']];
            if (in_array('Product', $types, true)) {
                return $node;
            }
        }
        foreach ($node as $v) {
            if (is_array($v)) {
                $stack[] = $v;
            }
        }
    }
    return null;
}

/**
 * Parse all <script type="application/ld+json"> and find first Product node.
 */
function sb_parse_schema_product($html) {
    $schemaProduct = null;
    $schemaRaw     = null;

    if (!preg_match_all('~<script[^>]+type=["\']application/ld\+json["\'][^>]*>(.*?)</script>~is', $html, $matches)) {
        return [null, null];
    }

    foreach ($matches[1] as $block) {
        $json = trim(html_entity_decode($block, ENT_QUOTES | ENT_HTML5, 'UTF-8'));
        if ($json === '') {
            continue;
        }

        // Some sites end JSON with ; at the end
        $json = preg_replace('~;\s*$~', '', $json);

        $data = json_decode($json, true);
        if (!is_array($data)) {
            continue;
        }

        $candidate = sb_find_product_in_jsonld($data);
        if ($candidate) {
            $schemaProduct = $candidate;
            $schemaRaw     = $json;
            break;
        }
    }

    return [$schemaProduct, $schemaRaw];
}

/**
 * Create DOM + XPath
 */
function sb_dom_xpath($html) {
    $dom = new DOMDocument();
    libxml_use_internal_errors(true);
    $dom->loadHTML($html);
    libxml_clear_errors();
    $xpath = new DOMXPath($dom);
    return [$dom, $xpath];
}

function sb_inner_html(DOMNode $node) {
    $doc = $node->ownerDocument;
    $html = '';
    foreach ($node->childNodes as $child) {
        $html .= $doc->saveHTML($child);
    }
    return $html;
}

function sb_text(DOMNode $node) {
    return trim(preg_replace('/\s+/u', ' ', $node->textContent));
}

/**
 * Extract descriptions, specs, variants from DOM.
 */
function sb_parse_dom_data($html) {
    list($dom, $xpath) = sb_dom_xpath($html);

    // Short description block
    $shortHtml = null;
    $nodes = $xpath->query("//div[contains(@class,'product-description-short')]");
    if ($nodes && $nodes->length > 0) {
        $shortHtml = sb_inner_html($nodes->item(0));
    }

    // Full description column (tabs + long text)
    $fullHtml = null;
    $nodes = $xpath->query("//div[contains(@class,'product_desc_column')]");
    if ($nodes && $nodes->length > 0) {
        $fullHtml = sb_inner_html($nodes->item(0));
    }

    // Specs: data-sheet + kfafeaturespro-table
    $features = [];
    $specContainers = $xpath->query("//div[contains(@class,'data-sheet') and contains(@class,'kfafeaturespro-table')]");
    if ($specContainers && $specContainers->length > 0) {
        foreach ($specContainers as $container) {
            $groupTitle = null;
            $h3 = $xpath->query("preceding-sibling::h3[1]", $container);
            if ($h3 && $h3->length > 0) {
                $groupTitle = sb_text($h3->item(0));
            }
            if (!$groupTitle) {
                $groupTitle = 'مشخصات';
            }

            $items = [];
            $dls = $xpath->query(".//dl", $container);
            foreach ($dls as $dl) {
                $dt = $xpath->query(".//dt", $dl)->item(0);
                $dd = $xpath->query(".//dd", $dl)->item(0);
                if (!$dt || !$dd) continue;

                $label = sb_text($dt);
                $value = sb_text($dd);

                if ($label === '' && $value === '') continue;

                $items[] = [
                    'label' => $label,
                    'value' => $value,
                ];
            }

            if (!empty($items)) {
                $features[] = [
                    'group' => $groupTitle,
                    'items' => $items,
                ];
            }
        }
    }

    // Variants: .product-variants-item → select options
    $variants = [];
    $variantBlocks = $xpath->query("//div[contains(@class,'product-variants-item')]");
    if ($variantBlocks && $variantBlocks->length > 0) {
        foreach ($variantBlocks as $block) {
            $labelNode = $xpath->query(".//span[contains(@class,'control-label')]", $block)->item(0);
            $label = $labelNode ? sb_text($labelNode) : null;

            $select = $xpath->query(".//select", $block)->item(0);
            if (!$select) continue;

            $attrName = $select->getAttribute('name');
            $options  = [];
            foreach ($xpath->query(".//option[not(@disabled)]", $select) as $opt) {
                /** @var DOMElement $opt */
                $val   = $opt->getAttribute('value');
                $text  = sb_text($opt);
                if ($text === '') continue;

                $options[] = [
                    'value'    => $val,
                    'label'    => $text,
                    'selected' => $opt->hasAttribute('selected'),
                ];
            }

            if (!empty($options)) {
                $variants[] = [
                    'attribute_name' => $attrName,
                    'label'          => $label,
                    'options'        => $options,
                ];
            }
        }
    }

    return [
        'description_short_html' => $shortHtml,
        'description_full_html'  => $fullHtml,
        'features'               => $features,
        'variants'               => $variants,
    ];
}

/* ===================== Main ===================== */

$url = isset($_GET['url']) ? $_GET['url'] : null;
if (!$url) {
    echo json_encode([
        'success' => false,
        'error'   => 'Missing ?url= parameter',
    ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit;
}

list($html, $err) = sb_fetch_url($url);
if ($html === null) {
    echo json_encode([
        'success' => false,
        'error'   => $err,
    ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit;
}

// 1) JSON-LD Product
list($schemaProduct, $schemaRaw) = sb_parse_schema_product($html);

// 2) DOM based parse
$domData = sb_parse_dom_data($html);

// 3) Meta fallbacks (title, description, og:image, product:price)
list($dom, $xpath) = sb_dom_xpath($html);

$metaTitle = null;
$metaDesc  = null;
$ogImage   = null;
$metaPrice = null;
$metaCurr  = null;

$nodes = $xpath->query("//title");
if ($nodes && $nodes->length > 0) {
    $metaTitle = sb_text($nodes->item(0));
}
$nodes = $xpath->query("//meta[@name='description']/@content");
if ($nodes && $nodes->length > 0) {
    $metaDesc = trim($nodes->item(0)->nodeValue);
}
$nodes = $xpath->query("//meta[@property='og:image']/@content");
if ($nodes && $nodes->length > 0) {
    $ogImage = trim($nodes->item(0)->nodeValue);
}
$nodes = $xpath->query("//meta[@property='product:price:amount']/@content");
if ($nodes && $nodes->length > 0) {
    $metaPrice = trim($nodes->item(0)->nodeValue);
}
$nodes = $xpath->query("//meta[@property='product:price:currency']/@content");
if ($nodes && $nodes->length > 0) {
    $metaCurr = trim($nodes->item(0)->nodeValue);
}

// 4) Build product array

$product = [
    'name'        => null,
    'sku'         => null,
    'mpn'         => null,
    'brand'       => null,
    'url'         => $url,
    'price'       => null,
    'currency'    => null,
    'availability'=> null,
    'images'      => [],
    'rating'      => null,
    'reviews'     => [],
    'description_short_html' => $domData['description_short_html'],
    'description_full_html'  => $domData['description_full_html'],
    'features'    => $domData['features'],
    'variants'    => $domData['variants'],
];

// Fill from JSON-LD if available
if ($schemaProduct) {
    $product['name'] = sb_safe_get($schemaProduct, 'name', $product['name']);
    $product['url']  = sb_safe_get($schemaProduct, 'url',  $product['url']);
    $product['sku']  = sb_safe_get($schemaProduct, 'sku',  $product['sku']);
    $product['mpn']  = sb_safe_get($schemaProduct, 'mpn',  $product['mpn']);

    $brand = sb_safe_get($schemaProduct, 'brand', null);
    if (is_array($brand)) {
        $product['brand'] = sb_safe_get($brand, 'name', null);
    } else {
        $product['brand'] = $brand;
    }

    $offers = sb_safe_get($schemaProduct, 'offers', null);
    if (is_array($offers)) {
        $product['price']       = sb_safe_get($offers, 'price', null);
        $product['currency']    = sb_safe_get($offers, 'priceCurrency', null);
        $product['availability']= sb_safe_get($offers, 'availability', null);
    }

    $images = sb_safe_get($schemaProduct, 'image', []);
    if (!is_array($images)) {
        $images = $images ? [$images] : [];
    }
    $product['images'] = $images;

    $agg = sb_safe_get($schemaProduct, 'aggregateRating', null);
    if (is_array($agg)) {
        $product['rating'] = [
            'ratingValue' => sb_safe_get($agg, 'ratingValue', null),
            'reviewCount' => sb_safe_get($agg, 'reviewCount', null),
            'ratingCount' => sb_safe_get($agg, 'ratingCount', null),
            'bestRating'  => sb_safe_get($agg, 'bestRating', null),
            'worstRating' => sb_safe_get($agg, 'worstRating', null),
        ];
    }

    $reviews = sb_safe_get($schemaProduct, 'review', []);
    if (!is_array($reviews)) {
        $reviews = [];
    } elseif (isset($reviews['@type'])) {
        $reviews = [$reviews];
    }
    foreach ($reviews as $r) {
        if (!is_array($r)) continue;
        $author = sb_safe_get($r, 'author', null);
        if (is_array($author)) {
            $author = sb_safe_get($author, 'name', null);
        }
        $ratingVal = null;
        if (isset($r['reviewRating']) && is_array($r['reviewRating'])) {
            $ratingVal = sb_safe_get($r['reviewRating'], 'ratingValue', null);
        }
        $product['reviews'][] = [
            'author'        => $author,
            'datePublished' => sb_safe_get($r, 'datePublished', null),
            'ratingValue'   => $ratingVal,
            'reviewBody'    => sb_safe_get($r, 'reviewBody', sb_safe_get($r, 'description', null)),
        ];
    }
}

// Fallbacks if JSON-LD did not provide price/currency/images
if ($product['price'] === null && $metaPrice !== null) {
    $product['price'] = $metaPrice;
}
if ($product['currency'] === null && $metaCurr !== null) {
    $product['currency'] = $metaCurr;
}
if (empty($product['images']) && $ogImage) {
    $product['images'] = [$ogImage];
}

$meta = [
    'title'       => $metaTitle,
    'description' => $metaDesc,
    'og_image'    => $ogImage,
];

// Final JSON
echo json_encode([
    'success'    => true,
    'source_url' => $url,
    'product'    => $product,
    'meta'       => $meta,
    // 'schema_raw' => $schemaRaw, // uncomment for debugging
], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
