<?php
/**
 * Simple category scraper for mobomobo.ir
 * Input:  GET url, pages, delay_ms
 * Output: JSON (names + absolute product URLs)
 *
 * Notes:
 * - از DOMXPath برای استخراج schema.org/Product استفاده می‌کنیم.
 * - نام از itemprop="name" و لینک از itemprop="url" گرفته می‌شود.
 * - لینک‌ها به آدرس مطلق تبدیل می‌شوند.
 * - صفحه‌ی اول خودِ URL است؛ صفحات بعدی با ?page=N یا &page=N ساخته می‌شوند.
 */

mb_internal_encoding('UTF-8');
header('Content-Type: application/json; charset=utf-8');

$inputUrl  = isset($_GET['url']) ? trim($_GET['url']) : '';
$totalPages = isset($_GET['pages']) ? (int)$_GET['pages'] : 1;
$delayMs   = isset($_GET['delay_ms']) ? max(0, (int)$_GET['delay_ms']) : 300;

if ($inputUrl === '') {
    http_response_code(400);
    echo json_encode([
        'ok' => false,
        'error' => 'Missing required parameter: url',
    ], JSON_UNESCAPED_UNICODE);
    exit;
}
if ($totalPages < 1) $totalPages = 1;

/** Fetch a URL with cURL */
function fetch($url) {
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_MAXREDIRS => 5,
        CURLOPT_CONNECTTIMEOUT => 10,
        CURLOPT_TIMEOUT => 20,
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_SSL_VERIFYHOST => 2,
        CURLOPT_HTTPHEADER => [
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
            'Accept-Language: fa-IR,fa;q=0.9,en-US;q=0.8,en;q=0.7',
            'Cache-Control: no-cache',
        ],
        CURLOPT_USERAGENT => 'CategoryScraper/1.0 (+https://example.com)',
    ]);
    $html = curl_exec($ch);
    $err  = curl_error($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($html === false || $code >= 400) {
        return [null, $err ?: "HTTP $code"];
    }
    return [$html, null];
}

/** Build page URL by setting/overriding ?page=N */
function withPageParam($url, $pageNum) {
    $parts = parse_url($url);
    $scheme = $parts['scheme'] ?? 'https';
    $host   = $parts['host']   ?? '';
    $port   = isset($parts['port']) ? ':' . $parts['port'] : '';
    $path   = $parts['path']   ?? '';
    $query  = [];
    if (!empty($parts['query'])) {
        parse_str($parts['query'], $query);
    }
    if ($pageNum > 1) {
        $query['page'] = $pageNum;
    } else {
        // صفحه ۱: پارامتر page را حذف می‌کنیم تا URL تمیز بماند
        unset($query['page']);
    }
    $queryStr = http_build_query($query);
    $final = $scheme . '://' . $host . $port . $path;
    if ($queryStr !== '') $final .= '?' . $queryStr;
    return $final;
}

/** Convert relative href to absolute */
function absolutize($baseUrl, $href) {
    // اگر همین‌جا absolute بود
    if (preg_match('~^https?://~i', $href)) return $href;

    $b = parse_url($baseUrl);
    $scheme = $b['scheme'] ?? 'https';
    $host   = $b['host'] ?? '';
    $port   = isset($b['port']) ? ':' . $b['port'] : '';
    $basePath = $b['path'] ?? '/';

    // Root-relative
    if (strpos($href, '/') === 0) {
        return $scheme . '://' . $host . $port . $href;
    }

    // Path-relative
    $dir = rtrim(dirname($basePath), '/\\') . '/';
    return $scheme . '://' . $host . $port . $dir . $href;
}

/** Parse product cards using schema.org/Product */
function parseProducts($html, $baseUrl) {
    $doc = new DOMDocument();
    libxml_use_internal_errors(true);
    $loaded = $doc->loadHTML($html, LIBXML_NOERROR | LIBXML_NOWARNING | LIBXML_NOCDATA);
    libxml_clear_errors();
    if (!$loaded) return [];

    $xp = new DOMXPath($doc);

    // هر اسکپ محصول که itemtype شامل Product باشد
    $productNodes = $xp->query("//*[(contains(@itemtype,'schema.org/Product') or contains(@itemtype,'Product')) and (@itemscope or @itemprop)]");

    $items = [];
    foreach ($productNodes as $node) {
        // نام محصول
        $nameNode = $xp->query(".//*[@itemprop='name' or self::h3 or self::h2]", $node)->item(0);
        $name = $nameNode ? trim($nameNode->textContent) : null;

        // لینک محصول (itemprop=url یا نزدیک‌ترین <a> داخل کارت)
        $urlAttr = $xp->query(".//*[@itemprop='url']/@href", $node)->item(0);
        if ($urlAttr) {
            $href = $urlAttr->nodeValue;
        } else {
            $a = $xp->query(".//a[@href][1]", $node)->item(0);
            $href = $a ? $a->getAttribute('href') : null;
        }

        if ($href) {
            $href = absolutize($baseUrl, $href);
        }

        if ($name && $href) {
            $items[] = [
                'name' => preg_replace('/\s+/u', ' ', $name),
                'url'  => $href,
            ];
        }
    }

    // اگر ساختار محصول متفاوت بود، fallback:
    if (empty($items)) {
        // کارت‌هایی که عنوان و لینک دارند (fallback عمومی)
        $links = $xp->query("//a[@href]");
        foreach ($links as $a) {
            $title = trim($a->textContent);
            $href  = $a->getAttribute('href');
            if ($title !== '' && $href !== '') {
                $items[] = [
                    'name' => preg_replace('/\s+/u', ' ', $title),
                    'url'  => absolutize($baseUrl, $href),
                ];
            }
        }
    }

    return $items;
}

/** Deduplicate by URL */
function dedupByUrl($arr) {
    $seen = [];
    $out = [];
    foreach ($arr as $it) {
        $key = strtolower($it['url']);
        if (!isset($seen[$key])) {
            $seen[$key] = true;
            $out[] = $it;
        }
    }
    return $out;
}

/** Sleep in ms */
function sleepMs($ms) {
    if ($ms <= 0) return;
    usleep($ms * 1000);
}

// ---------- Main ----------
$all = [];
$errors = [];
for ($p = 1; $p <= $totalPages; $p++) {
    $pageUrl = withPageParam($inputUrl, $p);
    list($html, $err) = fetch($pageUrl);
    if ($err) {
        $errors[] = ['page' => $p, 'url' => $pageUrl, 'error' => $err];
        continue;
    }
    $items = parseProducts($html, $pageUrl);
    $all = array_merge($all, $items);
    sleepMs($delayMs);
}

$all = dedupByUrl($all);

echo json_encode([
    'ok' => true,
    'input' => [
        'url' => $inputUrl,
        'pages_requested' => $totalPages,
        'delay_ms' => $delayMs,
    ],
    'meta' => [
        'count' => count($all),
        'generated_at' => gmdate('c'),
    ],
    'products' => $all,
    'errors' => $errors, // اگر صفحه‌ای خطا داد، این‌جا می‌بینی
], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);