<?php
/**
 * Product Details Web Service (JSON)
 * by: you
 *
 * Usage:
 *   /product_api.php?url=https://mobomobo.ir/2072
 *
 * Output:
 * {
 *   "ok": true,
 *   "meta": { "fetched_at": "...", "source_url": "..." },
 *   "product": {
 *     "url": "...",
 *     "name": "...",
 *     "description": "...",
 *     "image_main": "...",
 *     "images": ["...", "..."],
 *     "base_price": 228000,
 *     "currency": null,
 *     "variants": [
 *       { "id": "4471", "name": "iPhone 16Pro", "price": 228000, "stock": 13 },
 *       ...
 *     ]
 *   }
 * }
 */

mb_internal_encoding('UTF-8');
header('Content-Type: application/json; charset=utf-8');

// ---------- Input ----------
$inputUrl = isset($_GET['url']) ? trim($_GET['url']) : '';
if ($inputUrl === '') {
    http_response_code(400);
    echo json_encode(['ok' => false, 'error' => 'Missing required parameter: url'], JSON_UNESCAPED_UNICODE);
    exit;
}

// ---------- HTTP Fetch ----------
function fetch_html($url) {
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_MAXREDIRS => 5,
        CURLOPT_CONNECTTIMEOUT => 12,
        CURLOPT_TIMEOUT => 25,
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_SSL_VERIFYHOST => 2,
        CURLOPT_USERAGENT => 'ProductDetailsAPI/1.1 (+https://example.com)',
        CURLOPT_HTTPHEADER => [
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
            'Accept-Language: fa-IR,fa;q=0.9,en-US;q=0.8,en;q=0.7',
            'Cache-Control: no-cache',
        ],
    ]);
    $html = curl_exec($ch);
    $err  = curl_error($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($html === false || $code >= 400) {
        return [null, $err ?: "HTTP $code"];
    }
    return [$html, null];
}

// ---------- Helpers ----------
function absolutize($baseUrl, $href) {
    if (!$href) return null;
    $href = trim($href);
    if ($href === '') return null;

    if (preg_match('~^https?://~i', $href)) return $href;
    if (strpos($href, '//') === 0) {
        $b = parse_url($baseUrl);
        $scheme = $b['scheme'] ?? 'https';
        return $scheme . ':' . $href;
    }

    $b = parse_url($baseUrl);
    $scheme = $b['scheme'] ?? 'https';
    $host   = $b['host'] ?? '';
    $port   = isset($b['port']) ? ':' . $b['port'] : '';
    $basePath = $b['path'] ?? '/';

    if (strpos($href, '/') === 0) {
        return $scheme . '://' . $host . $port . $href;
    }

    $dir = rtrim(dirname($basePath), '/\\') . '/';
    return $scheme . '://' . $host . $port . $dir . $href;
}
function text_or_null($node) {
    if (!$node) return null;
    $t = trim(preg_replace('/\s+/u', ' ', $node->textContent));
    return $t !== '' ? $t : null;
}
function push_unique(&$arr, $val) {
    if ($val === null) return;
    if (!in_array($val, $arr, true)) $arr[] = $val;
}

// ---------- Parser ----------
function parse_product($html, $baseUrl) {
    $doc = new DOMDocument();
    libxml_use_internal_errors(true);
    $loaded = $doc->loadHTML($html, LIBXML_NOERROR | LIBXML_NOWARNING | LIBXML_NOCDATA);
    libxml_clear_errors();
    if (!$loaded) return ['ok' => false, 'error' => 'Failed to parse HTML'];

    $xp = new DOMXPath($doc);

    // ---- NAME (robust, avoids breadcrumb "خانه") ----
    $name = null;

    // Preferred: <h1> that is NOT inside breadcrumb/nav
    $h1Nodes = $xp->query("//h1[not(ancestor::*[contains(@class,'breadcrumb') or self::nav or @role='navigation'])]");
    foreach ($h1Nodes as $h1) {
        $txt = text_or_null($h1);
        if ($txt && mb_strlen($txt) > 3 && $txt !== 'خانه') { $name = $txt; break; }
    }

    // Next: itemprop="name" inside Product, but skip breadcrumb and "خانه"
    if (!$name) {
        $nodes = $xp->query("//*[contains(@itemtype,'schema.org/Product')]//*[@itemprop='name']");
        foreach ($nodes as $n) {
            // skip if this node is inside breadcrumb
            $isBreadcrumb = $xp->evaluate("boolean(ancestor::*[contains(@class,'breadcrumb') or self::nav or @role='navigation'])", $n);
            if ($isBreadcrumb) continue;
            $txt = text_or_null($n);
            if ($txt && mb_strlen($txt) > 3 && $txt !== 'خانه') { $name = $txt; break; }
        }
    }

    // Fallback: og:title
    if (!$name) {
        $ogt = $xp->query("//meta[@property='og:title']/@content")->item(0);
        $name = $ogt ? trim($ogt->nodeValue) : null;
    }

    // ---- DESCRIPTION ----
    $description = null;
    $d1 = $xp->query("//*[contains(@itemtype,'schema.org/Product')]//*[@itemprop='description']")->item(0);
    $description = text_or_null($d1);
    if (!$description) {
        $m1 = $xp->query("//meta[@name='description']/@content")->item(0);
        if ($m1) $description = trim($m1->nodeValue);
    }
    if (!$description) {
        $ogd = $xp->query("//meta[@property='og:description']/@content")->item(0);
        if ($ogd) $description = trim($ogd->nodeValue);
    }

    // ---- IMAGES ----
    $images = [];
    // main from og:image
    $image_main = null;
    $ogimg = $xp->query("//meta[@property='og:image']/@content")->item(0);
    if ($ogimg) $image_main = absolutize($baseUrl, trim($ogimg->nodeValue));
    if ($image_main) push_unique($images, $image_main);

    // itemprop=image & generic galleries (owl-carousel/thumb/gallery)
    foreach ($xp->query("//*[contains(@itemtype,'schema.org/Product')]//*[@itemprop='image' or self::img]") as $img) {
        $src = null;
        if ($img->hasAttribute('content')) $src = $img->getAttribute('content');
        if (!$src && $img->hasAttribute('src')) $src = $img->getAttribute('src');
        if (!$src && $img->hasAttribute('data-src')) $src = $img->getAttribute('data-src');
        $src = absolutize($baseUrl, $src);
        if ($src) push_unique($images, $src);
    }
    foreach ($xp->query("//div[contains(@class,'gallery') or contains(@class,'carousel') or contains(@class,'thumb') ]//img") as $img) {
        $src = $img->getAttribute('src') ?: $img->getAttribute('data-src');
        $src = absolutize($baseUrl, $src);
        if ($src) push_unique($images, $src);
    }

    // ---- VARIANTS (select > option[data-price|data-stock]) ----
    $variants = [];
    foreach ($xp->query("//select//option[@value and (@data-price or @data-stock or @data-amount)]") as $opt) {
        $v = [
            'id'    => $opt->getAttribute('value') ?: null,
            'name'  => text_or_null($opt),
            'price' => null,
            'stock' => null,
        ];
        if ($opt->hasAttribute('data-price') && $opt->getAttribute('data-price') !== '') {
            $v['price'] = (float)$opt->getAttribute('data-price');
        } elseif ($opt->hasAttribute('data-amount') && $opt->getAttribute('data-amount') !== '') {
            $v['price'] = (float)$opt->getAttribute('data-amount');
        }
        if ($opt->hasAttribute('data-stock') && $opt->getAttribute('data-stock') !== '') {
            $v['stock'] = (int)$opt->getAttribute('data-stock');
        }
        // فیلتر گزینه‌های بی‌نام یا بی‌ارزش
        if ($v['name'] && mb_strlen($v['name']) > 1) {
            $variants[] = $v;
        }
    }

    // ---- BASE PRICE (fallbacks) ----
    $base_price = null;
    // 1) data-price روی هر عنصر
    foreach ($xp->query("//*[@data-price]") as $el) {
        $val = $el->getAttribute('data-price');
        if (is_numeric($val)) { $base_price = (float)$val; break; }
    }
    // 2) eval="collect(XXXX, ...)"
    if ($base_price === null) {
        foreach ($xp->query("//*/@eval") as $attr) {
            if (preg_match('/collect\(\s*([0-9]+(?:\.[0-9]+)?)\s*,/u', $attr->nodeValue, $m)) {
                $base_price = (float)$m[1]; break;
            }
        }
    }

    // ---- RESULT ----
    return [
        'ok' => true,
        'product' => [
            'url'         => $baseUrl,
            'name'        => $name,
            'description' => $description,
            'image_main'  => $image_main,
            'images'      => array_values($images),
            'base_price'  => $base_price,  // ممکن است null باشد
            'currency'    => null,         // اگر همیشه تومان است، می‌توانید "IRT" بدهید
            'variants'    => $variants,
        ],
    ];
}

// ---------- RUN ----------
list($html, $err) = fetch_html($inputUrl);
if ($err) {
    http_response_code(502);
    echo json_encode(['ok' => false, 'error' => $err, 'url' => $inputUrl], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES|JSON_PRETTY_PRINT);
    exit;
}

$result = parse_product($html, $inputUrl);

// ---------- Output ----------
echo json_encode([
    'ok'      => $result['ok'],
    'meta'    => [
        'fetched_at' => gmdate('c'),
        'source_url' => $inputUrl,
    ],
    'product' => $result['product'] ?? null,
], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);