<?php
declare(strict_types=1);
header('Content-Type: application/json; charset=utf-8');
error_reporting(E_ALL);
ini_set('display_errors', '0');
set_time_limit(60);

/** ---------- helpers ---------- **/
function respond($status, array $payload) {
  http_response_code($status);
  echo json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
  exit;
}
function httpGet($url, $timeout = 25) {
  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_MAXREDIRS      => 5,
    CURLOPT_CONNECTTIMEOUT => 12,
    CURLOPT_TIMEOUT        => $timeout,
    CURLOPT_USERAGENT      => 'Mozilla/5.0 (compatible; ProductDetailsScraper/1.5)',
    CURLOPT_HTTPHEADER     => [
      'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
      'Accept-Language: fa,en;q=0.9'
    ],
    CURLOPT_ENCODING       => '' // gzip/deflate
  ]);
  $body = curl_exec($ch);
  $err  = curl_error($ch);
  $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);
  return [$code, $body, $err];
}
function textify($node) {
  return trim(preg_replace('/\s+/u', ' ', is_string($node) ? $node : $node->textContent));
}
function arr($x){ return is_array($x) ? $x : ($x!==null ? [$x] : []); }

/** normalize any color string (hex3/hex6/rgb/rgba) -> lowercase hex6 like #aabbcc */
function normalizeColorCode($c) {
  $c = trim(str_ireplace('!important','', $c));
  // hex
  if (preg_match('/^#([0-9a-f]{3})$/i', $c, $m)) {
    $h = strtolower($m[1]);
    return '#' . $h[0].$h[0] . $h[1].$h[1] . $h[2].$h[2];
  }
  if (preg_match('/^#([0-9a-f]{6})$/i', $c, $m)) {
    return '#' . strtolower($m[1]);
  }
  // rgb/rgba
  if (preg_match('/rgba?\s*\(\s*([0-9]{1,3})\s*,\s*([0-9]{1,3})\s*,\s*([0-9]{1,3})/i', $c, $m)) {
    $r = max(0, min(255, (int)$m[1]));
    $g = max(0, min(255, (int)$m[2]));
    $b = max(0, min(255, (int)$m[3]));
    return sprintf('#%02x%02x%02x', $r, $g, $b);
  }
  // named colors (rare) – return as-is
  return strtolower($c);
}

/** ---------- JSON-LD ---------- **/
function extractJsonLd($html) {
  $out = [];
  if (preg_match_all('#<script[^>]+type=["\']application/ld\+json["\'][^>]*>(.*?)</script>#is', $html, $m)) {
    foreach ($m[1] as $raw) {
      $raw = trim(preg_replace('#/\*.*?\*/#s','',$raw));
      $json = json_decode($raw, true);
      if ($json === null) $json = json_decode(preg_replace('/[\x00-\x1F]/','',$raw), true);
      if ($json !== null) $out[] = $json;
    }
  }
  return $out;
}
function parseFromJsonLd($html) {
  $name = null; $price = null; $currency = null; $images = []; $desc = null; $rating = null; $reviewCount = null;

  foreach (extractJsonLd($html) as $blk) {
    if (isset($blk['@graph']) && is_array($blk['@graph'])) {
      foreach ($blk['@graph'] as $g) {
        $tp = isset($g['@type']) ? (is_string($g['@type']) ? $g['@type'] : implode(',', (array)$g['@type'])) : '';
        if (stripos($tp, 'Product') !== false) {
          $name   = $name ?: ($g['name'] ?? null);
          $desc   = $desc ?: ($g['description'] ?? null);
          $images = array_values(array_unique(array_filter(array_map('strval', arr($g['image'] ?? [])))));
          if (isset($g['offers'])) {
            $offers = $g['offers'];
            if (isset($offers['price']))         $price    = $offers['price'];
            if (isset($offers['priceCurrency'])) $currency = $offers['priceCurrency'];
          }
          if (isset($g['aggregateRating'])) {
            $rating = $g['aggregateRating']['ratingValue'] ?? null;
            $reviewCount = $g['aggregateRating']['reviewCount'] ?? null;
          }
        }
      }
    }
    if (isset($blk['@type'])) {
      $tp = is_string($blk['@type']) ? $blk['@type'] : implode(',', (array)$blk['@type']);
      if (stripos($tp, 'Product') !== false) {
        $name   = $name ?: ($blk['name'] ?? null);
        $desc   = $desc ?: ($blk['description'] ?? null);
        $images = array_values(array_unique(array_filter(array_map('strval', arr($blk['image'] ?? [])))));
        if (isset($blk['offers'])) {
          $offers = $blk['offers'];
          if (isset($offers['price']))         $price    = $offers['price'];
          if (isset($offers['priceCurrency'])) $currency = $offers['priceCurrency'];
        }
        if (isset($blk['aggregateRating'])) {
          $rating = $blk['aggregateRating']['ratingValue'] ?? null;
          $reviewCount = $blk['aggregateRating']['reviewCount'] ?? null;
        }
      }
    }
  }
  return [$name, $price, $currency, $images, $desc, $rating, $reviewCount];
}

/** ---------- DOM fallbacks ---------- **/
function parseFromDom($html) {
  $dom = new DOMDocument(); @$dom->loadHTML($html, LIBXML_NOERROR | LIBXML_NOWARNING);
  $xp  = new DOMXPath($dom);

  $name = null; $desc = null; $images = []; $price = null;

  $h = $xp->query("//h1 | //h2[contains(@class,'title')]"); if ($h && $h->length) $name = textify($h->item(0));

  $ds = $xp->query("//section[contains(@class,'pro-detail')] | //div[contains(@class,'product-description')]");
  if ($ds && $ds->length) $desc = trim(preg_replace('/\s+/u',' ', $ds->item(0)->textContent));
  if (!$desc) {
    if (preg_match('#<meta[^>]+name=["\']description["\'][^>]+content=["\']([^"\']+)#i', $html, $m)) $desc = trim($m[1]);
    if (!$desc && preg_match('#<meta[^>]+property=["\']og:description["\'][^>]+content=["\']([^"\']+)#i', $html, $m2)) $desc = trim($m2[1]);
  }

  if (preg_match('#<meta[^>]+property=["\']og:image["\'][^>]+content=["\']([^"\']+)#i', $html, $m)) $images[] = $m[1];
  foreach (['//img[contains(@class,"product") or contains(@class,"zoom") or contains(@class,"gallery")]', '//div[contains(@class,"gallery")]//img'] as $q) {
    $nodes = $xp->query($q);
    if ($nodes) foreach ($nodes as $img) {
      $src = $img->getAttribute('data-src') ?: $img->getAttribute('src');
      if ($src) $images[] = $src;
    }
  }
  $images = array_values(array_unique(array_filter($images)));

  if (preg_match('/([\d,]+)\s*تومان/u', $html, $m)) $price = (int)str_replace(',', '', $m[1]);

  return [$name, $price, 'IRR', $images, $desc];
}

/** ---------- Models ---------- **/
function extractOptionsByGroupTitle($html, $groupKeyword) {
  $dom = new DOMDocument(); @$dom->loadHTML($html, LIBXML_NOERROR | LIBXML_NOWARNING);
  $xp  = new DOMXPath($dom);
  $results = [];
  $boxes = $xp->query(
    "//*[contains(@class,'lbl-opt') or contains(@class,'options') or self::section or self::div]
      [ .//*[contains(normalize-space(.), '$groupKeyword')] ]"
  );
  if ($boxes && $boxes->length) {
    foreach ($boxes as $box) {
      $nodes = (new DOMXPath($dom))->query(".//label[contains(@class,'option')]//span | .//label[@for]//span | .//label[@for]", $box);
      if ($nodes) foreach ($nodes as $n) {
        $t = textify($n);
        if ($t === '') {
          $lab = $n instanceof DOMElement ? $n : $n->parentNode;
          if ($lab instanceof DOMElement) {
            $t = trim($lab->getAttribute('title') ?: $lab->getAttribute('data-title') ?: $lab->getAttribute('aria-label') ?: $lab->getAttribute('data-bs-original-title'));
          }
        }
        if ($t !== '' && !in_array($t, $results, true)) $results[] = $t;
      }
    }
  }
  return $results;
}
function parseModels($html) {
  $models = extractOptionsByGroupTitle($html, 'مدل');
  if (empty($models)) $models = extractOptionsByGroupTitle($html, 'مدل گوشی');
  if (empty($models)) $models = extractOptionsByGroupTitle($html, 'Model');
  if (empty($models)) {
    $dom = new DOMDocument(); @$dom->loadHTML($html, LIBXML_NOERROR | LIBXML_NOWARNING);
    $xp  = new DOMXPath($dom);
    $nodes = $xp->query("//label[starts-with(@for,'option-5-') or starts-with(@for,'option-7-')]/span");
    if ($nodes) foreach ($nodes as $n) {
      $t = textify($n);
      if ($t !== '' && !in_array($t, $models, true)) $models[] = $t;
    }
  }
  $models = array_values(array_unique($models));
  natcasesort($models);
  return array_values($models);
}

/** ---------- Colors: map 50+ popular codes to Persian names ---------- **/
function colorNameMap() {
  // کلیدها hex شش‌رقمیِ lowercase هستند
  return [
    '#000000'=>'مشکی',
    '#ffffff'=>'سفید',
    '#808080'=>'خاکستری',
    '#a9a9a9'=>'خاکستری تیره',
    '#d3d3d3'=>'خاکستری روشن',
    '#c0c0c0'=>'نقره‌ای',
    '#ffd700'=>'طلایی',
    '#daa520'=>'طلایی تیره',
    '#8b4513'=>'قهوه‌ای تیره',
    '#a0522d'=>'قهوه‌ای',
    '#d2b48c'=>'قهوه‌ای روشن / تان',
    '#f5f5dc'=>'بژ / کرم',
    '#dbb28e'=>'بژ / دیزرت',
    '#ff00ff'=>'سرخابی',
    '#ff1493'=>'صورتی پررنگ',
    '#ffc0cb'=>'صورتی',
    '#ff69b4'=>'صورتی متوسط',
    '#ff99aa'=>'صورتی روشن',
    '#800080'=>'بنفش',
    '#9932cc'=>'بنفش متوسط',
    '#8a2be2'=>'آبی بنفش',
    '#4b0082'=>'نیلی',
    '#000080'=>'سرمه‌ای',
    '#0000ff'=>'آبی',
    '#1e90ff'=>'آبی دوجر',
    '#87cefa'=>'آبی آسمانی روشن',
    '#87ceeb'=>'آبی آسمانی',
    '#00bfff'=>'آبی آسمانی تیره',
    '#4169e1'=>'آبی سلطنتی',
    '#00ffff'=>'فیروزه‌ای / آبی فیروزه‌ای',
    '#40e0d0'=>'فیروزه‌ای روشن',
    '#20b2aa'=>'آکوامارین تیره',
    '#008080'=>'سبزآبی / تیل',
    '#008000'=>'سبز',
    '#006400'=>'سبز تیره',
    '#228b22'=>'سبز جنگلی',
    '#32cd32'=>'سبز لایم',
    '#7fff00'=>'سبز لیمویی',
    '#98fb98'=>'سبز کمرنگ',
    '#adff2f'=>'سبز زرد',
    '#ffff00'=>'زرد',
    '#ffd700'=>'طلایی',          // تکرار آگاهانه برای اطمینان
    '#ffa500'=>'نارنجی',
    '#ff8c00'=>'نارنجی تیره',
    '#ff7f50'=>'نارنجی مرجانی',
    '#ff4500'=>'نارنجی قرمز',
    '#ff6347'=>'گوجه‌ای',
    '#ff0000'=>'قرمز',
    '#dc143c'=>'قرمز کرمزی',
    '#b22222'=>'قرمز آجری',
    '#8b0000'=>'زرشکی',
    '#800000'=>'عنابی',
    '#2f4f4f'=>'خاکستری زغالی',
    '#708090'=>'خاکستری مایل به آبی',
    '#ffe4c4'=>'بیسکویتی',
    '#faf0e6'=>'کتانی (لینن)',
    '#f0e68c'=>'خاکی',
    '#e6e6fa'=>'یاسی',
    '#f0ffff'=>'آزور',
    '#f5fffa'=>'نعنایی',
    '#fdf5e6'=>'کرم استخوانی'
  ];
}

function parseColors($html) {
  $dom = new DOMDocument(); @$dom->loadHTML($html, LIBXML_NOERROR | LIBXML_NOWARNING);
  $xp  = new DOMXPath($dom);

  $map = colorNameMap();
  $colors = [];              // خروجی نهایی: نام‌های فارسی
  $selectedColorName = '';   // نام رنگ انتخاب‌شده کنار «رنگ: ...»

  // 0) نام رنگ انتخاب‌شده (اگر در صفحه رندر شده باشد)
  $sel = $xp->query("//*[contains(normalize-space(.),'رنگ')]/following::*[self::span or self::b][1]");
  if ($sel && $sel->length) {
    $candidate = textify($sel->item(0));
    if ($candidate !== '' && mb_strpos($candidate, 'رنگ') === false) $selectedColorName = $candidate;
  }

  // 1) color-box: دایره‌های رنگ
  $groups = $xp->query("//*[contains(normalize-space(.),'رنگ :') or contains(normalize-space(.),'رنگ:')]/following::div[contains(@class,'color-box')][1]");
  if ($groups && $groups->length) {
    $box = $groups->item(0);
    $labels = (new DOMXPath($dom))->query(".//label", $box);
    foreach ($labels as $lab) {
      /** @var DOMElement $lab */
      $name = ''; $code = '';
      $sp = $lab->getElementsByTagName('span');
      if ($sp->length) {
        $style = $sp->item(0)->getAttribute('style');
        if ($style && preg_match('/background(?:-color)?\s*:\s*([^;]+)/i', $style, $m)) {
          $code = normalizeColorCode($m[1]);
        }
        // اگر روی span/label نامی بود
        $name = trim(
          $sp->item(0)->getAttribute('title') ?: $sp->item(0)->getAttribute('data-title') ?: ''
        );
      }
      if ($name === '') {
        $name = trim(
          $lab->getAttribute('title') ?: $lab->getAttribute('data-title') ?: $lab->getAttribute('aria-label') ?: $lab->getAttribute('data-bs-original-title')
        );
      }
      // اگر نام نداریم ولی کد داریم، از مپ به نام تبدیل کن
      if ($name === '' && $code !== '') {
        if (isset($map[$code])) $name = $map[$code];
      }
      // اگر هنوز هم خالی است، خود کد رنگ را بده که خروجی خالی نشود
      if ($name === '' && $code !== '') $name = $code;

      if ($name !== '' && !in_array($name, $colors, true)) $colors[] = $name;
    }
  }

  // 2) fallback: هر عنصر با background-color (اگر color-box پیدا نشد)
  if (empty($colors)) {
    $nodes = $xp->query("//*[self::span or self::i][contains(translate(@style,'ABCDEFGHIJKLMNOPQRSTUVWXYZ','abcdefghijklmnopqrstuvwxyz'),'background-color')]");
    foreach ($nodes as $n) {
      $style = $n->getAttribute('style');
      if ($style && preg_match('/background(?:-color)?\s*:\s*([^;]+)/i', $style, $m)) {
        $code = normalizeColorCode($m[1]);
        $name = isset($map[$code]) ? $map[$code] : $code;
        if (!in_array($name, $colors, true)) $colors[] = $name;
      }
    }
  }

  return [$colors, $selectedColorName];
}

/** ---------- controller ---------- **/
$url = isset($_GET['url']) ? trim($_GET['url']) : '';
if ($url === '') respond(400, ['error' => 'Missing required parameter: url']);
if (!filter_var($url, FILTER_VALIDATE_URL)) respond(400, ['error' => 'Invalid URL']);

list($code, $html, $err) = httpGet($url);
if ($err || $code >= 400 || !$html) {
  respond(502, ['error' => $err ?: ('HTTP ' . $code), 'url' => $url]);
}

list($name, $priceIRR, $currency, $images, $desc, $rating, $reviewCount) = parseFromJsonLd($html);
if (!$name || !$priceIRR || empty($images) || !$desc) {
  list($name2, $price2, $cur2, $img2, $desc2) = parseFromDom($html);
  if (!$name && $name2) $name = $name2;
  if (!$priceIRR && $price2) $priceIRR = $price2 * 10; // اگر DOM تومان بود
  if (empty($images) && !empty($img2)) $images = $img2;
  if (!$desc && $desc2) $desc = $desc2;
  if (!$currency && $cur2) $currency = $cur2;
}

$models = parseModels($html);
list($colors, $selectedColorName) = parseColors($html);

// تومان = price(IRR) / 10
$priceToman = null;
if ($priceIRR !== null) {
  $n = is_numeric($priceIRR) ? (float)$priceIRR : (float)str_replace(',', '', (string)$priceIRR);
  $priceToman = (int)round($n / 10);
}

$out = [
  'url'         => $url,
  'name'        => $name ?: '',
  'price'       => $priceToman,
  'currency'    => 'TOMAN',
  'images'      => array_values($images ?: []),
  'models'      => array_values($models ?: []),
  'colors'      => array_values($colors ?: []),           // نام فارسی یا در نهایت خود کُد
  'selected_color_name' => $selectedColorName ?: '',
  'description' => $desc ? trim($desc) : ''
];
if ($rating !== null || $reviewCount !== null) $out['rating'] = ['value' => $rating, 'count' => $reviewCount];

respond(200, $out);