<?php
/**
 * category_api.php
 * وب‌سرویس: با ورودی‌های زیر، نام و لینک تمام محصولات دسته‌بندی را (در چند صفحه) برمی‌گرداند.
 *   GET:
 *     - category_url=  آدرس دسته‌بندی (می‌تواند فارسی باشد، مثل: https://exo.ir/category/فروش-قطعات-کامپیوتر)
 *     - pages=         تعداد صفحات برای پیمایش (≥1، پیش‌فرض 1، سقف 100)
 *
 * خروجی:
 *   {
 *     "ok": true,
 *     "input": { "category_url": "...", "pages_requested": N },
 *     "fetched": [ { "url": "...", "http_code": 200, "content_type":"...", "bytes":12345, "error": null }, ... ],
 *     "count": <تعداد محصولات یکتا>,
 *     "products": [ { "name": "...", "url": "..." }, ... ]
 *   }
 */

declare(strict_types=1);
mb_internal_encoding('UTF-8');

/* -------------------- Utils -------------------- */
function respond_json($data, int $code = 200): void {
    http_response_code($code);
    header('Content-Type: application/json; charset=utf-8');
    header('X-Content-Type-Options: nosniff');
    echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
    exit;
}
function fail(string $msg, int $code = 400, array $extra = []): void {
    respond_json(['ok' => false, 'error' => $msg] + $extra, $code);
}

/* -------------------- URL helpers (Unicode-safe) -------------------- */
// نرمال‌سازی URL با پشتیبانی از دامنه/مسیر فارسی
function normalize_url_unicode(string $url): string {
    $url = trim($url);
    if ($url === '') return $url;

    // حذف فاصله‌های ناخواسته
    $url = preg_replace('/\s+/u', '', $url);

    $p = parse_url($url);
    if (!$p || empty($p['host'])) return $url;

    // دامنه → Punycode (اگر intl فعال باشد)
    $host = $p['host'];
    if (function_exists('idn_to_ascii')) {
        $puny = idn_to_ascii($host, IDNA_DEFAULT, INTL_IDNA_VARIANT_UTS46);
        if ($puny) $host = $puny;
    }

    // مسیر → encode segment-by-segment (اجتناب از دوباره‌کدکردن)
    $path = $p['path'] ?? '';
    if ($path !== '') {
        $segments = array_map('rawurldecode', explode('/', $path));
        $segments = array_map('rawurlencode', $segments);
        $path = implode('/', $segments);
        if ($path === '' || $path[0] !== '/') $path = '/'.$path;
    }

    // کوئری → encode key/value
    $query = '';
    if (!empty($p['query'])) {
        parse_str($p['query'], $qarr);
        $pairs = [];
        foreach ($qarr as $k => $v) {
            $pairs[] = rawurlencode((string)$k) . '=' . rawurlencode((string)$v);
        }
        $query = $pairs ? '?'.implode('&', $pairs) : '';
    }

    $scheme = $p['scheme'] ?? 'https';
    $port   = isset($p['port']) ? ':'.$p['port'] : '';
    $frag   = isset($p['fragment']) ? '#'.rawurlencode($p['fragment']) : '';

    return "{$scheme}://{$host}{$port}{$path}{$query}{$frag}";
}

// افزودن/به‌روزرسانی پارامترهای کوئری (سازگار با URL فارسی)
function set_query_arg_unicode(string $url, string $key, string $value): string {
    $url = normalize_url_unicode($url);
    $p = parse_url($url);
    if (!$p) return $url;
    $scheme = $p['scheme'] ?? 'https';
    $host   = $p['host']   ?? '';
    $port   = isset($p['port']) ? ':'.$p['port'] : '';
    $path   = $p['path']   ?? '';
    $queryA = [];
    if (!empty($p['query'])) parse_str($p['query'], $queryA);
    $queryA[$key] = $value;
    $pairs = [];
    foreach ($queryA as $k => $v) $pairs[] = rawurlencode((string)$k) . '=' . rawurlencode((string)$v);
    $qs  = $pairs ? '?'.implode('&', $pairs) : '';
    $frag = isset($p['fragment']) ? '#'.rawurlencode($p['fragment']) : '';
    return "{$scheme}://{$host}{$port}{$path}{$qs}{$frag}";
}

// ساخت URL مطلق از نسبی (برای ساخت لینک‌های محصول)
function abs_url(string $base, string $url): string {
    if (!$url) return '';
    if (preg_match('#^https?://#i', $url)) return $url;
    $base = normalize_url_unicode($base);
    $bp = parse_url($base);
    $scheme = $bp['scheme'] ?? 'https';
    $host   = $bp['host']   ?? '';
    if (str_starts_with($url, '//')) return $scheme.':'.$url;
    if ($url[0] === '/') return $scheme.'://'.$host.$url;
    $dir = rtrim(preg_replace('#/[^/]*$#', '/', $bp['path'] ?? '/'), '/').'/';
    return $scheme.'://'.$host.$dir.$url;
}

/* -------------------- HTTP -------------------- */
function fetch_html(string $url, int $timeout = 20): array {
    $url = normalize_url_unicode($url);
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_MAXREDIRS => 5,
        CURLOPT_CONNECTTIMEOUT => 10,
        CURLOPT_TIMEOUT => $timeout,
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_SSL_VERIFYHOST => 2,
        CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 '
          .'(KHTML, like Gecko) Chrome/127.0 Safari/537.36 (+category_api.php)',
        CURLOPT_HTTPHEADER => [
            'Accept: text/html,application/xhtml+xml;q=0.9,*/*;q=0.8',
            'Accept-Language: fa-IR,fa;q=0.9,en;q=0.8',
        ],
        CURLOPT_ENCODING => '', // gzip/deflate/br
    ]);
    $body = curl_exec($ch);
    $err  = curl_error($ch);
    $info = curl_getinfo($ch);
    curl_close($ch);
    return [$body, $err, $info];
}

/* -------------------- DOM -------------------- */
function load_xp(string $html): DOMXPath {
    libxml_use_internal_errors(true);
    $dom = new DOMDocument();
    @$dom->loadHTML('<?xml encoding="utf-8"?>'.$html, LIBXML_NOWARNING|LIBXML_NOERROR|LIBXML_COMPACT);
    return new DOMXPath($dom);
}
function txt(?DOMNode $n): string {
    if (!$n) return '';
    return trim(preg_replace('/\s+/u', ' ', $n->textContent ?? ''));
}

/* -------------------- Parser: Extract product cards -------------------- */
/**
 * تلاش برای استخراج کارت‌های محصول از صفحهٔ دسته‌بندی.
 * - الگوی اختصاصی exo.ir: div.grid-product → a[href*="/product/"] + عنوان h3/h5
 * - الگوی عمومی fallback: هر a[href*="/product/"] با تایتل/متن معنی‌دار
 */
function extract_category_products(string $pageUrl, DOMXPath $xp): array {
    $found = [];

    // الگوی اصلی exo.ir
    $nodes = $xp->query('//div[contains(@class,"grid-product")]//a[contains(@href,"/product/")]');
    if ($nodes && $nodes->length) {
        $docXP = new DOMXPath($xp->document);
        foreach ($nodes as $a) {
            $href = abs_url($pageUrl, $a->getAttribute('href'));
            // پیدا کردن عنوان نزدیک
            $title = trim($a->getAttribute('title'));
            if ($title === '') {
                $parent = $a->parentNode;
                $h = null;
                foreach (['.//h3','.//h5','.//div[contains(@class,"product-title")]','.//p'] as $rel) {
                    $h = $docXP->query($rel, $parent)->item(0);
                    if ($h && ($t = txt($h))) { $title = $t; break; }
                }
            }
            if ($title === '') $title = txt($a);
            if ($href && $title && mb_strlen($title) >= 3) {
                $found[] = ['name'=>$title, 'url'=>$href];
            }
        }
    }

    // fallback عمومی
    if (!$found) {
        foreach ($xp->query('//a[contains(@href,"/product/")]') as $a) {
            $href = abs_url($pageUrl, $a->getAttribute('href'));
            $title = trim($a->getAttribute('title')) ?: txt($a);
            if ($href && $title && mb_strlen($title) >= 3) {
                $found[] = ['name'=>$title, 'url'=>$href];
            }
        }
    }

    // dedupe
    $seen = [];
    $unique = [];
    foreach ($found as $p) {
        if (!isset($seen[$p['url']])) {
            $seen[$p['url']] = true;
            $unique[] = $p;
        }
    }
    return $unique;
}

/* -------------------- MAIN -------------------- */
$categoryUrl = $_GET['category_url'] ?? '';
$pagesParam  = $_GET['pages'] ?? '1';

if (!$categoryUrl) fail('پارامتر ?category_url= الزامی است.');

// نرمال‌سازی برای URL فارسی
$categoryUrl = normalize_url_unicode($categoryUrl);

// اعتبارسنجی منعطف: اگر FILTER_VALIDATE_URL رد کرد، parse_url را چک کن
if (!filter_var($categoryUrl, FILTER_VALIDATE_URL)) {
    $p = parse_url($categoryUrl);
    if (!$p || empty($p['scheme']) || empty($p['host'])) {
        fail('آدرس دسته‌بندی نامعتبر است.', 422, ['given' => $_GET['category_url']]);
    }
}

$pages = (int)$pagesParam;
if ($pages < 1) $pages = 1;
if ($pages > 100) $pages = 100; // سقف منطقی

$all = [];
$fetched = [];

for ($i = 1; $i <= $pages; $i++) {
    $pageUrl = $categoryUrl;
    if ($i > 1) $pageUrl = set_query_arg_unicode($categoryUrl, 'page', (string)$i);

    [$body, $err, $info] = fetch_html($pageUrl);
    $fetched[] = [
        'url' => $pageUrl,
        'http_code' => $info['http_code'] ?? null,
        'content_type' => $info['content_type'] ?? null,
        'bytes' => $body ? strlen($body) : 0,
        'error' => $err ?: null,
    ];

    if ($err || !$body || (($info['http_code'] ?? 0) >= 400)) {
        // اگر این صفحه خطا داد، ادامه می‌دهیم تا صفحات بعدی هم امتحان شوند
        continue;
    }

    $xp = load_xp($body);
    $items = extract_category_products($info['url'] ?? $pageUrl, $xp);

    // اگر در صفحات بعدی، هیچ آیتمی پیدا نشد، حدس می‌زنیم انتهای صفحه‌بندی است و حلقه را قطع می‌کنیم
    if ($i > 1 && !$items) break;

    foreach ($items as $it) $all[] = $it;
}

// dedupe نهایی
$map = [];
$result = [];
foreach ($all as $p) {
    if (!isset($map[$p['url']])) {
        $map[$p['url']] = true;
        $result[] = $p;
    }
}

// خروجی
respond_json([
    'ok' => true,
    'input' => [
        'category_url' => $categoryUrl,
        'pages_requested' => $pages,
    ],
    'fetched' => $fetched,
    'count' => count($result),
    'products' => $result,
]);