<?php
/**
 * Digikala Product Proxy (PHP 7+)
 *
 * مثال استفاده:
 *   digikala.php?utl=https://www.digikala.com/product/dkp-17816092/...
 *   digikala.php?url=https://www.digikala.com/product/dkp-17816092/...
 *   digikala.php?id=17816092
 */

declare(strict_types=1);

// هدرهای خروجی
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// ---- helper: خروجی خطا به صورت JSON ----
function dg_proxy_error(string $msg, int $status = 400, array $extra = []) : void {
    http_response_code($status);
    echo json_encode(array_merge([
        'ok'      => false,
        'error'   => $msg,
        'status'  => $status,
    ], $extra), JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
    exit;
}

// ---- گرفتن ورودی‌ها (utl | url | id) ----
$utlInput = isset($_GET['utl']) ? trim((string)$_GET['utl']) : '';
$urlInput = isset($_GET['url']) ? trim((string)$_GET['url']) : '';
$idInput  = isset($_GET['id'])  ? trim((string)$_GET['id'])  : '';

$dkp    = null;
$source = null;

// 1) اگر id مستقیم داده شده
if ($idInput !== '' && ctype_digit($idInput)) {
    $dkp    = $idInput;
    $source = 'id';
}

// 2) اگر utl/url داده شده
if ($dkp === null) {
    $url = $utlInput !== '' ? $utlInput : $urlInput;
    if ($url !== '') {
        // استخراج dkp-123456
        if (preg_match('~dkp-(\d+)~i', $url, $m)) {
            $dkp    = $m[1];
            $source = 'url';
        } elseif (preg_match('~[?&]dkp=(\d+)~i', $url, $m)) {
            $dkp    = $m[1];
            $source = 'url';
        }
    }
}

// اگر هنوز dkp مشخص نشده
if ($dkp === null) {
    dg_proxy_error(
        'پارامتر utL/url یا id معتبر نیست. نمونه: ?utl=https://www.digikala.com/product/dkp-17816092/... یا ?id=17816092',
        400
    );
}

// ---- درخواست به API دیجی‌کالا ----
$endpoint = "https://api.digikala.com/v2/product/{$dkp}/";

$ch = curl_init();
curl_setopt_array($ch, [
    CURLOPT_URL            => $endpoint,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_MAXREDIRS      => 5,
    CURLOPT_CONNECTTIMEOUT => 10,
    CURLOPT_TIMEOUT        => 25,
    CURLOPT_HTTPHEADER     => [
        'Accept: application/json, text/plain, */*',
        'User-Agent: Mozilla/5.0 (Proxy/1.0; PHP)',
        'Referer: https://www.digikala.com/',
    ],
]);

$body     = curl_exec($ch);
$curlErr  = curl_error($ch);
$httpCode = (int) curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

if ($body === false || $curlErr) {
    dg_proxy_error(
        'خطا در اتصال به دیجی‌کالا: ' . $curlErr,
        502,
        ['endpoint' => $endpoint]
    );
}

if ($httpCode !== 200) {
    $snippet = mb_substr(strip_tags((string)$body), 0, 200, 'UTF-8');
    dg_proxy_error(
        'پاسخ نامعتبر از دیجی‌کالا (HTTP ' . $httpCode . ')',
        502,
        [
            'endpoint' => $endpoint,
            'http_code'=> $httpCode,
            'snippet'  => $snippet,
        ]
    );
}

// ---- decode JSON اصلی ----
$data = json_decode($body, true);
if (!is_array($data) || empty($data['data']['product'])) {
    dg_proxy_error(
        'ساختار پاسخ دیجی‌کالا قابل تشخیص نیست.',
        502,
        ['endpoint' => $endpoint]
    );
}

$p = $data['data']['product'];

// ---- helper برای تصویر اصلی ----
$mainImage = null;
if (!empty($p['images']['main']['url'][0])) {
    $mainImage = $p['images']['main']['url'][0];
} elseif (!empty($p['images']['main'][0]['url'])) {
    $mainImage = $p['images']['main'][0]['url'];
}
if ($mainImage !== null) {
    $mainImage = filter_var($mainImage, FILTER_VALIDATE_URL) ? $mainImage : null;
}

// ---- لیست تصاویر ----
$gallery = [];
if (!empty($p['images']['list']) && is_array($p['images']['list'])) {
    foreach ($p['images']['list'] as $img) {
        $u = '';
        if (!empty($img['url'][0])) {
            $u = $img['url'][0];
        } elseif (!empty($img['url'])) {
            $u = is_array($img['url']) ? reset($img['url']) : $img['url'];
        }
        $u = trim((string)$u);
        if ($u !== '' && filter_var($u, FILTER_VALIDATE_URL)) {
            $gallery[] = $u;
        }
    }
}

// ---- خلاصهٔ قیمت‌ها از روی واریانت‌ها ----
$variantsRaw = !empty($p['variants']) && is_array($p['variants']) ? $p['variants'] : [];
$minSelling  = null;
$maxSelling  = null;
$minRrp      = null;
$maxRrp      = null;

foreach ($variantsRaw as $v) {
    if (empty($v['price']) || !is_array($v['price'])) {
        continue;
    }
    $sp = isset($v['price']['selling_price']) ? (float)$v['price']['selling_price'] : 0.0;
    $rp = isset($v['price']['rrp_price'])     ? (float)$v['price']['rrp_price']     : 0.0;

    if ($sp > 0) {
        $minSelling = ($minSelling === null) ? $sp : min($minSelling, $sp);
        $maxSelling = ($maxSelling === null) ? $sp : max($maxSelling, $sp);
    }
    if ($rp > 0) {
        $minRrp = ($minRrp === null) ? $rp : min($minRrp, $rp);
        $maxRrp = ($maxRrp === null) ? $rp : max($maxRrp, $rp);
    }
}

// ---- نرمال‌سازی واریانت‌ها (متغیرها) + اتریبیوت‌ها ----
$variants = [];
$ignoreVariantKeys = [
    'price','seller','marketplace_seller','marketplace','images',
    'is_incredible','is_incredible_notif','digiplus_eligible',
    'promotions','packing','shipment','dkp','dkpc','tracking',
    'rate','rating','votes','max_per_order','min_per_order',
];

foreach ($variantsRaw as $v) {
    if (!is_array($v)) continue;

    $vId     = isset($v['id']) ? (int)$v['id'] : null;
    $vStatus = isset($v['status']) ? (string)$v['status'] : '';
    $vPrice  = isset($v['price']) && is_array($v['price']) ? $v['price'] : [];

    $attrs = [];

    foreach ($v as $k => $val) {
        if ($k === 'id' || $k === 'status' || in_array($k, $ignoreVariantKeys, true)) {
            continue;
        }

        $key   = (string)$k;
        $label = $key;
        $value = null;

        if (is_array($val)) {
            // تلاش برای برداشتن label/value استاندارد
            $label = isset($val['label'])      ? (string)$val['label']      : $label;
            $label = isset($val['title_fa'])   ? (string)$val['title_fa']   : $label;
            $value = isset($val['value'])      ? (string)$val['value']      : $value;
            $value = isset($val['title_fa'])   ? (string)$val['title_fa']   : $value;
            $value = isset($val['title_en'])   ? (string)$val['title_en']   : $value;
        } else {
            $value = (string)$val;
        }

        if ($value === null || $value === '') {
            continue;
        }

        $attrs[$key] = [
            'label' => $label,
            'value' => $value,
        ];
    }

    $variants[] = [
        'id'       => $vId,
        'status'   => $vStatus,
        'price'    => [
            'selling_price' => isset($vPrice['selling_price']) ? (float)$vPrice['selling_price'] : null,
            'rrp_price'     => isset($vPrice['rrp_price'])     ? (float)$vPrice['rrp_price']     : null,
            'is_incredible' => isset($vPrice['is_incredible']) ? (bool)$vPrice['is_incredible']  : null,
        ],
        'attributes' => $attrs,
        'raw'        => $v, // کل واریانت برای مواقعی که همه‌چیز لازم است
    ];
}

// ---- مشخصات فنی (specifications) ----
$specGroups = [];
if (!empty($p['specifications']) && is_array($p['specifications'])) {
    foreach ($p['specifications'] as $group) {
        if (!is_array($group)) continue;

        $gTitle = isset($group['title']) ? (string)$group['title'] : '';
        $items  = [];

        if (!empty($group['attributes']) && is_array($group['attributes'])) {
            foreach ($group['attributes'] as $attr) {
                if (!is_array($attr)) continue;
                $aTitle  = isset($attr['title'])  ? (string)$attr['title']  : '';
                $aValues = isset($attr['values']) && is_array($attr['values'])
                    ? array_values(array_map('strval', $attr['values']))
                    : [];

                if ($aTitle === '' || empty($aValues)) {
                    continue;
                }

                $items[] = [
                    'title'  => $aTitle,
                    'values' => $aValues,
                ];
            }
        }

        if ($gTitle !== '' && !empty($items)) {
            $specGroups[] = [
                'group' => $gTitle,
                'items' => $items,
            ];
        }
    }
}

// ---- ساخت آبجکت محصول نهایی ----
$product = [
    'id'       => isset($p['id']) ? (int)$p['id'] : (int)$dkp,
    'dkp'      => (int)$dkp,
    'title'    => [
        'fa' => isset($p['title_fa']) ? (string)$p['title_fa'] : '',
        'en' => isset($p['title_en']) ? (string)$p['title_en'] : '',
    ],
    'status'   => isset($p['status']) ? (string)$p['status'] : '',
    'url'      => isset($p['url']) ? (string)$p['url'] : '',
    'category' => [
        'id'    => isset($p['category']['id'])        ? (int)$p['category']['id'] : null,
        'title' => isset($p['category']['title_fa'])  ? (string)$p['category']['title_fa'] : null,
    ],
    'brand'    => [
        'id'    => isset($p['brand']['id'])        ? (int)$p['brand']['id'] : null,
        'title' => isset($p['brand']['title_fa'])  ? (string)$p['brand']['title_fa'] : null,
    ],
    'rating'   => [
        'rate'  => isset($p['rating']['rate'])  ? (float)$p['rating']['rate']  : null,
        'count' => isset($p['rating']['count']) ? (int)$p['rating']['count']   : null,
    ],
    'price_summary' => [
        'currency'        => 'IRR',
        'min_selling'     => $minSelling,
        'max_selling'     => $maxSelling,
        'min_rrp'         => $minRrp,
        'max_rrp'         => $maxRrp,
    ],
    'images' => [
        'main' => $mainImage,
        'list' => $gallery,
    ],
    'descriptions' => [
        'short'         => isset($p['short_description']) ? (string)$p['short_description'] : '',
        'review_html'   => isset($p['review']['description']) ? (string)$p['review']['description'] : '',
        'expert_review' => isset($p['expert_reviews']) ? $p['expert_reviews'] : null,
    ],
    'variants'       => $variants,
    'specifications' => $specGroups,
    'raw_product'    => $p, // کل product خام دیجی‌کالا (برای اینکه "همه‌چیز" در دسترس باشد)
];

// ---- خروجی نهایی ----
$response = [
    'ok'       => true,
    'source'   => 'digikala',
    'dkp'      => (int)$dkp,
    'from'     => $source,
    'endpoint' => $endpoint,
    'product'  => $product,
];

echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);