<?php
/**
 * Simple Woo Category Scraper API (PHP 7 compatible)
 * GET:
 *   url (required) — category URL
 *   pg  (optional) — page number (default 1). Appends /page/{pg}/
 *   debug (optional) — 1 to include diagnostics
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');

// ---- helpers (PHP 7 safe) ----
function starts_with($haystack, $needle) {
    return $needle === '' || strpos($haystack, $needle) === 0;
}
function ends_with($haystack, $needle) {
    if ($needle === '') return true;
    return substr($haystack, -strlen($needle)) === $needle;
}
function json_error($message, $code = 400, $extra = []) {
    if (!headers_sent()) http_response_code($code);
    echo json_encode(array_merge(['ok' => false, 'error' => $message], $extra), JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
    exit;
}
// تبدیل خطاها به Exception تا صفحه سفید نشود
set_error_handler(function($severity, $message, $file, $line) {
    throw new ErrorException($message, 0, $severity, $file, $line);
});

$debug = isset($_GET['debug']) && $_GET['debug'] == '1';

try {
    $url = isset($_GET['url']) ? trim($_GET['url']) : '';
    $pg  = isset($_GET['pg'])  ? (int)$_GET['pg'] : 1;

    if ($url === '') json_error('پارامتر url الزامی است.');
    if (!preg_match('~^https?://~i', $url)) json_error('url باید با http(s) شروع شود.');
    if ($pg < 1) $pg = 1;

    if (!class_exists('DOMDocument')) {
        json_error('DOM extension روی سرور فعال نیست.', 500, ['fix' => 'enable php-xml / dom']);
    }

    // نرمال‌سازی کتگوری
    $rtrim = rtrim($url);
    $norm  = preg_replace('~(/page/\d+/?)$~i', '/', $rtrim);
    if (!ends_with($norm, '/')) $norm .= '/';

    // ساخت URL صفحه
    $target = ($pg === 1) ? $norm : rtrim($norm, '/') . '/page/' . $pg . '/';

    // دریافت HTML (cURL یا fallback)
    $html = '';
    $http_info = [];
    $curl_err = null;

    $ua = 'Mozilla/5.0 (compatible; CategoryScraper/1.1; +https://example.test)';
    if (function_exists('curl_init')) {
        $ch = curl_init($target);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_MAXREDIRS      => 5,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_TIMEOUT        => 20,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_SSL_VERIFYHOST => 2,
            CURLOPT_USERAGENT      => $ua,
            CURLOPT_HTTPHEADER     => ['Accept: text/html,application/xhtml+xml'],
        ]);
        $html = curl_exec($ch);
        $info = curl_getinfo($ch);
        $curl_err = curl_error($ch);
        curl_close($ch);
        $http_info = $info ?: [];
    } else {
        // fallback
        $context = stream_context_create([
            'http' => [
                'method' => 'GET',
                'header' => "User-Agent: $ua\r\nAccept: text/html,application/xhtml+xml\r\n",
                'timeout' => 20,
            ],
            'ssl' => [
                'verify_peer' => true,
                'verify_peer_name' => true,
                'allow_self_signed' => false
            ]
        ]);
        $html = @file_get_contents($target, false, $context);
        $http_info = ['note' => 'curl_disabled_fallback'];
    }

    if ($html === false || $html === '' || $html === null) {
        json_error('عدم دریافت محتوا از آدرس هدف.', 502, ['curl_error' => $curl_err, 'target' => $target]);
    }

    libxml_use_internal_errors(true);
    $dom = new DOMDocument();
    $loaded = $dom->loadHTML('<?xml encoding="utf-8" ?>' . $html, LIBXML_NOWARNING | LIBXML_NOERROR);
    libxml_clear_errors();
    if (!$loaded) json_error('خطا در تجزیه HTML.', 500);

    $xpath = new DOMXPath($dom);

    // انتخابگرهای ووکامرس
    $nodes = $xpath->query("//li[contains(concat(' ', normalize-space(@class), ' '), ' product ')]");
    $products = [];
    $seen = [];

    // تبدیل نسبی به کامل
    $bp = parse_url($target);
    $scheme = isset($bp['scheme']) ? $bp['scheme'] : 'https';
    $host   = isset($bp['host']) ? $bp['host'] : '';
    $port   = isset($bp['port']) ? ':' . $bp['port'] : '';
    $path   = isset($bp['path']) ? $bp['path'] : '/';
    $dir    = preg_replace('~/[^/]*$~', '/', $path);

    foreach ($nodes as $li) {
        $local = new DOMXPath($li->ownerDocument);
        $aNode = null;

        $anchors = $local->query(".//a[contains(@class,'woocommerce-LoopProduct-link') or contains(@class,'woocommerce-loop-product__link')]", $li);
        if ($anchors && $anchors->length) {
            $aNode = $anchors->item(0);
        } else {
            $anchors = $local->query(".//a", $li);
            if ($anchors && $anchors->length) $aNode = $anchors->item(0);
        }

        $href = $aNode ? trim($aNode->getAttribute('href')) : '';
        if ($href !== '') {
            if (preg_match('~^https?://~i', $href)) {
                // ok
            } elseif (starts_with($href, '//')) {
                $href = $scheme . ':' . $href;
            } elseif (starts_with($href, '/')) {
                $href = $scheme . '://' . $host . $port . $href;
            } else {
                $href = $scheme . '://' . $host . $port . $dir . $href;
            }
        }

        // عنوان
        $h2 = $local->query(".//h2[contains(@class,'woocommerce-loop-product__title')]", $li);
        $title = '';
        if ($h2 && $h2->length) {
            $title = trim(preg_replace('/\s+/u', ' ', $h2->item(0)->textContent));
        } elseif ($aNode) {
            $title = trim($aNode->getAttribute('title'));
            if ($title === '') $title = trim(preg_replace('/\s+/u', ' ', $aNode->textContent));
        }

        if ($href === '' || $title === '') continue;
        if (isset($seen[$href])) continue;
        $seen[$href] = true;

        $products[] = ['name' => $title, 'url' => $href];
    }

    $out = [
        'ok'       => true,
        'category' => $norm,
        'page'     => $pg,
        'target'   => $target,
        'count'    => count($products),
        'products' => $products,
    ];

    if ($debug) {
        $out['debug'] = [
            'http' => $http_info,
            'php'  => [
                'version' => PHP_VERSION,
                'curl'    => function_exists('curl_init') ? 'enabled' : 'disabled',
                'dom'     => class_exists('DOMDocument') ? 'enabled' : 'missing',
            ]
        ];
    }

    echo json_encode($out, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);

} catch (Throwable $e) {
    // هر خطایی به JSON تبدیل شود تا صفحه سفید نشود
    json_error('Exception: ' . $e->getMessage(), 500);
}