<?php
/**
 * Merge API: دسته → جزئیات محصولات
 *
 * Usage:
 *   alo_merge.php
 *     ?cat_url=https://aloghesti.com/digital-product/cell-phone/
 *     &pages=1
 *     &detail_api=https://majazi.top/v2/alo.php
 *     &list_api=https://majazi.top/v2/alo_daste.php
 *     &concurrency=8
 *     &max_items=100
 *     &fields=name,brand,price,currency,images,variants,features,description
 *     &include_variants=1
 *     &include_features=1
 *
 * Output JSON:
 * {
 *   "ok": true,
 *   "fetched": {...},
 *   "count": N,
 *   "items": [
 *     {
 *       "name": "...",
 *       "url": "...",
 *       "detail": {
 *         "name": "...",
 *         "brand": "...",
 *         "price": 123,
 *         "currency": "IRR",
 *         "images": [...],
 *         "variants": [...],
 *         "features": {...},
 *         "description": "..."
 *       }
 *     },
 *     ...
 *   ]
 * }
 */

header('Content-Type: application/json; charset=UTF-8');
// header('Access-Control-Allow-Origin: *'); // درصورت نیاز باز کن

$catUrl       = isset($_GET['cat_url']) ? trim($_GET['cat_url']) : '';
$pages        = isset($_GET['pages']) ? (int)$_GET['pages'] : 1;
$detailApi    = isset($_GET['detail_api']) ? trim($_GET['detail_api']) : 'https://majazi.top/v2/alo.php';
$listApi      = isset($_GET['list_api'])   ? trim($_GET['list_api'])   : 'https://majazi.top/v2/alo_daste.php';
$concurrency  = isset($_GET['concurrency']) ? max(1, (int)$_GET['concurrency']) : 8;
$maxItems     = isset($_GET['max_items']) ? max(1, (int)$_GET['max_items']) : 200;

$fieldsParam  = isset($_GET['fields']) ? trim($_GET['fields']) : 'name,brand,price,currency,images,variants,features,description';
$fieldsWanted = array_filter(array_map('trim', explode(',', $fieldsParam)));

$includeVariants = isset($_GET['include_variants']) ? ($_GET['include_variants'] == '1') : true;
$includeFeatures = isset($_GET['include_features']) ? ($_GET['include_features'] == '1') : true;

if (!$catUrl || !filter_var($catUrl, FILTER_VALIDATE_URL)) {
  http_response_code(400);
  echo json_encode(['ok' => false, 'error' => 'Missing or invalid cat_url'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  exit;
}
if (!filter_var($detailApi, FILTER_VALIDATE_URL) || !filter_var($listApi, FILTER_VALIDATE_URL)) {
  http_response_code(400);
  echo json_encode(['ok' => false, 'error' => 'Invalid detail_api or list_api URL'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  exit;
}

try {
  // 1) محصولات دسته
  $listUrl = $listApi . '?' . http_build_query([
    'url'   => $catUrl,
    'pages' => max(1, $pages),
  ]);
  $listJson = http_get_json($listUrl, 25);
  if (!$listJson || empty($listJson['ok']) || empty($listJson['products'])) {
    throw new Exception('Failed to fetch category list');
  }

  $products = $listJson['products'];
  if (count($products) > $maxItems) {
    $products = array_slice($products, 0, $maxItems);
  }

  // 2) جزئیات برای هر محصول (curl_multi)
  $detailUrls = [];
  foreach ($products as $p) {
    if (empty($p['url'])) continue;
    $detailUrls[] = $detailApi . '?' . http_build_query(['url' => $p['url']]);
  }
  $details = multi_http_get_json($detailUrls, 30, $concurrency);

  // 3) ادغام
  $items = [];
  foreach ($products as $idx => $p) {
    $detail = $details[$idx] ?? null;
    $payload = null;

    if ($detail && !empty($detail['ok']) && !empty($detail['data']) && is_array($detail['data'])) {
      $payload = filter_detail_fields($detail['data'], $fieldsWanted, $includeVariants, $includeFeatures);
    }

    $items[] = [
      'name'   => fa_fix($p['name'] ?? ''),
      'url'    => $p['url'] ?? '',
      'detail' => $payload,
    ];
  }

  echo json_encode([
    'ok' => true,
    'fetched' => [
      'category'      => $catUrl,
      'pages'         => $pages,
      'items_in_list' => count($products),
      'merged_at'     => gmdate('c'),
      'detail_api'    => $detailApi,
      'list_api'      => $listApi
    ],
    'count' => count($items),
    'items' => $items
  ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);

} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['ok' => false, 'error' => $e->getMessage()], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  exit;
}

/* ===================== Helpers ===================== */

function http_get_json(string $url, int $timeout = 20): ?array {
  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_MAXREDIRS      => 4,
    CURLOPT_CONNECTTIMEOUT => min(10, $timeout),
    CURLOPT_TIMEOUT        => $timeout,
    CURLOPT_SSL_VERIFYPEER => true,
    CURLOPT_HTTPHEADER     => [
      'Accept: application/json,text/plain,*/*',
      'User-Agent: Mozilla/5.0 (MergeBot)'
    ]
  ]);
  $body = curl_exec($ch);
  $err  = curl_error($ch);
  $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);
  if ($err || $code >= 400 || !$body) return null;
  $data = json_decode($body, true);
  return is_array($data) ? $data : null;
}

function multi_http_get_json(array $urls, int $timeout = 25, int $concurrency = 8): array {
  $mh = curl_multi_init();
  $handles = [];
  $results = array_fill(0, count($urls), null);

  $active = 0; $idx = 0;

  // تابع سازندهٔ هندل
  $mk = function($url) use ($timeout) {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_FOLLOWLOCATION => true,
      CURLOPT_MAXREDIRS      => 4,
      CURLOPT_CONNECTTIMEOUT => min(10, $timeout),
      CURLOPT_TIMEOUT        => $timeout,
      CURLOPT_SSL_VERIFYPEER => true,
      CURLOPT_HTTPHEADER     => [
        'Accept: application/json,text/plain,*/*',
        'User-Agent: Mozilla/5.0 (MergeBot)'
      ]
    ]);
    return $ch;
  };

  // استارت اولیه
  for ($i = 0; $i < min($concurrency, count($urls)); $i++, $idx++) {
    $ch = $mk($urls[$idx]);
    $handles[$idx] = $ch;
    curl_multi_add_handle($mh, $ch);
  }

  do {
    $status = curl_multi_exec($mh, $active);
    if ($status > CURLM_OK) break;

    // نتایج آماده را بردار و هندل بعدی را جایگزین کن
    while ($info = curl_multi_info_read($mh)) {
      $doneHandle = $info['handle'];
      $doneIdx = array_search($doneHandle, $handles, true);
      if ($doneIdx !== false) {
        $body = curl_multi_getcontent($doneHandle);
        $err  = curl_error($doneHandle);
        $code = curl_getinfo($doneHandle, CURLINFO_HTTP_CODE);

        $results[$doneIdx] = (!$err && $code < 400 && $body) ? json_decode($body, true) : null;

        curl_multi_remove_handle($mh, $doneHandle);
        curl_close($doneHandle);
        unset($handles[$doneIdx]);

        // اگر هنوز URL باقی است، یکی دیگر اضافه کن
        if ($idx < count($urls)) {
          $ch = $mk($urls[$idx]);
          $handles[$idx] = $ch;
          curl_multi_add_handle($mh, $ch);
          $idx++;
        }
      }
    }

    if ($active) curl_multi_select($mh, 0.5);
  } while ($active || $handles);

  curl_multi_close($mh);
  return $results;
}

function filter_detail_fields(array $data, array $fieldsWanted, bool $withVariants, bool $withFeatures): array {
  $out = [];
  foreach ($fieldsWanted as $f) {
    if (!array_key_exists($f, $data)) continue;
    if ($f === 'variants' && !$withVariants) continue;
    if ($f === 'features' && !$withFeatures) continue;
    $out[$f] = $data[$f];
  }
  // اطمینان از وجود name/price اگر در fields نیامده‌اند
  if (!isset($out['name']) && isset($data['name'])) $out['name'] = $data['name'];
  if (!isset($out['price']) && isset($data['price'])) $out['price'] = $data['price'];
  if (!isset($out['currency']) && isset($data['currency'])) $out['currency'] = $data['currency'];
  return $out;
}

function fa_fix($s) {
  if ($s === null) return null;
  $s = html_entity_decode($s, ENT_QUOTES | ENT_HTML5, 'UTF-8');
  $s = strtr($s, ['ي'=>'ی','ك'=>'ک']);
  $s = preg_replace('/\s+/u',' ', trim($s));
  return $s;
}
