<?php
/**
 * TehranMelody single product scraper (OpenCart theme) – PHP 7.0 compatible
 * Usage:
 *   GET  ?url=https://www.tehranmelody.com/_______     (اختیاری: &debug=1)
 *   یا
 *   POST/GET ?html=BASE64_OR_RAW_SOURCE                 (اختیاری: &debug=1)
 *
 * Output JSON:
 *   name, url, brand, sku, categories[], images[],
 *   price_text, price, currency,
 *   description_html, description_text,
 *   highlights{ key=>val }, specs{ key=>val }
 */

header('Content-Type: application/json; charset=UTF-8');
mb_internal_encoding('UTF-8');

function fail($msg, $code=500){
    http_response_code($code);
    echo json_encode(array('ok'=>false,'error'=>$msg), JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT);
    exit;
}

function fetch_html($url){
    $ch = curl_init($url);
    curl_setopt_array($ch, array(
        CURLOPT_RETURNTRANSFER=>true,
        CURLOPT_FOLLOWLOCATION=>true,
        CURLOPT_MAXREDIRS=>5,
        CURLOPT_TIMEOUT=>25,
        CURLOPT_CONNECTTIMEOUT=>15,
        CURLOPT_ENCODING=>'',
        CURLOPT_SSL_VERIFYPEER=>true,
        CURLOPT_SSL_VERIFYHOST=>2,
        CURLOPT_HTTPHEADER=>array(
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome Safari/537.36',
            'Accept-Language: fa-IR,fa;q=0.9,en;q=0.8'
        ),
    ));
    $html = curl_exec($ch);
    $err  = curl_error($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    if ($html===false || $code>=400) fail("HTTP $code while fetching: $url".($err?" | $err":""), $code ? $code : 500);
    if (!mb_detect_encoding($html,'UTF-8',true)) $html = mb_convert_encoding($html,'UTF-8','auto');
    return $html;
}

function get_input_html(){
    $url   = isset($_GET['url']) ? trim($_GET['url']) : '';
    $html  = isset($_POST['html']) ? $_POST['html'] : (isset($_GET['html']) ? $_GET['html'] : '');
    $debug = !empty($_GET['debug']);

    if ($html !== '') {
        $decoded = base64_decode($html, true);
        if ($decoded !== false && strlen($decoded) > 200) $html = $decoded;
        return array($url, $html, $debug);
    }
    if ($url==='') fail("please provide 'url' or 'html'", 400);
    return array($url, fetch_html($url), $debug);
}

function dom_xpath($html){
    libxml_use_internal_errors(true);
    $dom = new DOMDocument();
    @$dom->loadHTML('<?xml encoding="UTF-8">'.$html);
    libxml_clear_errors();
    return array($dom, new DOMXPath($dom));
}

function xtext($xp,$ctx,$x){
    $n = $xp->query($x,$ctx)->item(0);
    return $n ? trim(preg_replace('/\s+/u',' ', $n->textContent)) : '';
}
function xattr($xp,$ctx,$x,$a){
    $n = $xp->query($x,$ctx)->item(0);
    return $n ? trim($n->getAttribute($a)) : '';
}

function parse_jsonld($html){
    $out = array();
    if (!preg_match_all('#<script[^>]+type=["\']application/ld\+json["\'][^>]*>(.*?)</script>#si',$html,$m)) return $out;
    foreach ($m[1] as $block) {
        $json = html_entity_decode($block, ENT_QUOTES|ENT_HTML5, 'UTF-8');
        $data = json_decode($json, true);
        if (!$data) continue;
        $nodes = array();
        if (isset($data['@type'])) $nodes[] = $data;
        if (!empty($data['@graph']) && is_array($data['@graph'])) $nodes = array_merge($nodes,$data['@graph']);
        foreach ($nodes as $n) {
            if (!is_array($n)) continue;
            if (isset($n['@type']) && stripos(is_array($n['@type'])?implode(',',$n['@type']):$n['@type'], 'Product')!==false){
                $p = array();
                if (isset($n['name'])) $p['name']=$n['name'];
                if (isset($n['sku']))  $p['sku']=$n['sku'];
                if (isset($n['brand'])) {
                    $p['brand'] = is_array($n['brand']) ? (isset($n['brand']['name'])?$n['brand']['name']:null) : $n['brand'];
                }
                $offers = isset($n['offers']) ? $n['offers'] : array();
                if (isset($offers['@type'])) $offers = array($offers);
                foreach ((array)$offers as $o){
                    if (!isset($p['price']) && isset($o['price'])) $p['price'] = $o['price'];
                    if (!isset($p['currency']) && isset($o['priceCurrency'])) $p['currency'] = $o['priceCurrency'];
                    if (!isset($p['availability']) && !empty($o['availability'])) $p['availability'] = $o['availability'];
                }
                $imgs = array();
                $imgArr = isset($n['image']) ? (array)$n['image'] : array();
                foreach ($imgArr as $im){ $imgs[] = is_array($im)?(isset($im['url'])?$im['url']:''):$im; }
                // پاکسازی
                $uimgs = array();
                foreach ($imgs as $im){ if ($im!=='' && !in_array($im,$uimgs,true)) $uimgs[]=$im; }
                if (!empty($uimgs)) $p['images'] = $uimgs;

                if (!empty($n['description'])) $p['description_html'] = $n['description'];

                foreach ($p as $k=>$v){ if ($v===null || $v==='') unset($p[$k]); }
                $out = array_merge($out, $p);
            }
        }
    }
    return $out;
}

// نرمال‌سازی قیمت (ارقام فارسی→انگلیسی، حذف واحد/جداکننده)
function normalize_price($txt){
    if ($txt==='' || $txt===null) return null;
    $fa = array('۰','۱','۲','۳','۴','۵','۶','۷','۸','۹','٬','،',',','٫','.',' تومان','تومان','ريال','ریال',' ');
    $en = array('0','1','2','3','4','5','6','7','8','9','','' ,'' ,'' ,'' ,'' ,''      ,''     ,''    ,''   ,'');
    $t = str_replace($fa,$en,$txt);
    $t = preg_replace('/[^\d]/','',$t);
    return $t!=='' ? $t : null;
}

function parse_tehranmelody($html, $baseUrl=null){
    list($dom,$xp) = dom_xpath($html);
    $root = $dom->documentElement;
    $res = array();

    // نام محصول (مثال: <h1 class="dir-ltr"><span>...</span></h1>)
    $name = xtext($xp,$root,"//h1[contains(@class,'dir-ltr')]/span");
    if ($name==='') $name = xtext($xp,$root,"//h1[contains(@class,'product-title')]//span | //h1[contains(@class,'product-title')]");
    if ($name!=='') $res['name']=$name;

    // قیمت (داخل باکس قیمت)
    $priceTxt = xtext($xp,$root,"//div[contains(@class,'product-price')]//*[contains(@class,'price')]");
    if ($priceTxt==='') $priceTxt = xtext($xp,$root,"//div[contains(@class,'price-self')]//*[contains(@class,'price')]");
    if ($priceTxt==='') $priceTxt = xtext($xp,$root,"//*[contains(@class,'price')][contains(.,'تومان')]");
    if ($priceTxt!=='') {
        $res['price_text'] = $priceTxt;
        $np = normalize_price($priceTxt);
        if ($np!==null) { $res['price'] = $np; $res['currency']='IRT'; }
    }

    // برند (لوگوی برند کنار عنوان)
    $brand = xattr($xp,$root,"//div[contains(@class,'brand-product')]//img",'alt');
    if ($brand==='') $brand = xtext($xp,$root,"//div[contains(@class,'brand-product')]//a");
    if ($brand!=='') $res['brand']=$brand;

    // دسته‌بندی‌ها از breadcrumb
    $cats = array();
    foreach($xp->query("//li[contains(@class,'breadcrumb-item')]//span[@itemprop='name']") as $s){
        $t = trim($s->textContent);
        if ($t!=='') $cats[]=$t;
    }
    if (!empty($cats)) $res['categories']=$cats;

    // توضیحات (باکس review/description)
    $descNode = $xp->query("//div[@id='product-review']//div[contains(@class,'product-review')]")->item(0);
    if (!$descNode) $descNode = $xp->query("//div[@id='product-review']")->item(0);
    if ($descNode){
        $htmlDesc = '';
        foreach ($descNode->childNodes as $child) $htmlDesc .= $dom->saveHTML($child);
        $res['description_html'] = trim($htmlDesc);
        $res['description_text'] = trim(preg_replace('/\s+/u',' ', $descNode->textContent));
    }

    // ویژگی‌های برجسته (fieldset.top-attributes)
    $highlights = array();
    foreach($xp->query("//fieldset[contains(@class,'top-attributes')]/div") as $d){
        $k = xtext($xp,$d,".//strong");
        $v = xtext($xp,$d,".//div");
        if ($k!=='' && $v!=='') $highlights[$k]=$v;
    }
    if (!empty($highlights)) $res['highlights']=$highlights;

    // جدول مشخصات (#property table > tr: td[1]=key, td[2]=val)
    $specs = array();
    foreach($xp->query("//div[@id='property']//table//tbody//tr") as $tr){
        $tds = $xp->query(".//td", $tr);
        if ($tds->length>=2) {
            $k = trim(preg_replace('/\s+/u',' ', $tds->item(0)->textContent));
            $v = trim(preg_replace('/\s+/u',' ', $tds->item(1)->textContent));
            if ($k!=='' && $v!=='') $specs[$k]=$v;
        }
    }
    if (!empty($specs)) $res['specs']=$specs;

    // گالری تصاویر (a[data-fancybox="preview"]) + fallback img
    $imgs = array();
    foreach($xp->query("//div[contains(@class,'imglist')]//a[@data-fancybox='preview']") as $a){
        $href = $a->getAttribute('href');
        if ($href && !in_array($href,$imgs,true)) $imgs[]=$href;
    }
    foreach($xp->query("//div[contains(@class,'imglist')]//img") as $im){
        $src = $im->getAttribute('src');
        if ($src && !in_array($src,$imgs,true)) $imgs[]=$src;
    }
    if (!empty($imgs)) $res['images']=$imgs;

    if ($baseUrl) $res['url']=$baseUrl;

    // پاکسازی فیلدهای خالی (فقط رشته‌ها)
    foreach ($res as $k=>$v){
        if (is_string($v) && $v==='') unset($res[$k]);
    }
    return $res;
}

try{
    list($url,$html,$debug) = get_input_html();
    $fromJsonLd = parse_jsonld($html);
    $fromDom    = parse_tehranmelody($html, $url ? $url : null);

    // ادغام: DOM ارجح؛ مقادیری که در DOM نیست از JSON-LD پر می‌شود
    $product = $fromDom;
    foreach ($fromJsonLd as $k=>$v){
        if (!isset($product[$k]) || $product[$k]==='' || $product[$k]===null) {
            $product[$k] = $v;
        }
    }

    // اگر قیمت عددی تهی بود و JSON-LD داشت
    if ((!isset($product['price']) || $product['price']==='') && isset($fromJsonLd['price'])) {
        $product['price'] = normalize_price((string)$fromJsonLd['price']);
        if (!isset($product['price_text'])) $product['price_text'] = (string)$fromJsonLd['price'];
        if (!isset($product['currency']) && isset($fromJsonLd['currency'])) $product['currency'] = $fromJsonLd['currency'];
        elseif (!isset($product['currency'])) $product['currency'] = 'IRT';
    }

    $out = array('ok'=>true, 'product'=>$product);
    if ($debug) $out['debug'] = array(
        'name_from_dom'=> isset($fromDom['name']) ? $fromDom['name'] : null,
        'price_text'   => isset($fromDom['price_text']) ? $fromDom['price_text'] : null
    );
    echo json_encode($out, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES|JSON_PRETTY_PRINT);
} catch (Exception $e){
    fail($e->getMessage());
}